package ru.yandex.qe.dispenser.api.v1.request;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.util.JsonSerializerBase;
import ru.yandex.qe.dispenser.api.v1.DiAmount;

/**
 * DTO to update actual quota usage values.
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonSerialize(using = DiActualQuotaUpdate.Serializer.class)
public final class DiActualQuotaUpdate {

    @Nullable
    private final String resourceKey;
    @Nullable
    private final String projectKey;
    @Nullable
    private final String quotaSpecKey;
    @Nullable
    private final DiAmount actual;
    @Nullable
    private final Set<String> segments;

    @JsonCreator
    public DiActualQuotaUpdate(
            @JsonProperty("resourceKey") @Nullable final String resourceKey,
            @JsonProperty("projectKey") @Nullable final String projectKey,
            @JsonProperty("quotaSpecKey") @Nullable final String quotaSpecKey,
            @JsonProperty("actual") @Nullable final DiAmount actual,
            @JsonProperty("segments") @Nullable final Set<String> segments) {
        this.resourceKey = resourceKey;
        this.projectKey = projectKey;
        this.quotaSpecKey = quotaSpecKey;
        this.actual = actual;
        this.segments = segments;
    }

    @NotNull
    public static Builder forResource(@NotNull final String resourceKey) {
        return new Builder(resourceKey);
    }

    @Nullable
    public String getResourceKey() {
        return resourceKey;
    }

    @Nullable
    public String getProjectKey() {
        return projectKey;
    }

    @Nullable
    public String getQuotaSpecKey() {
        return quotaSpecKey;
    }

    @Nullable
    public DiAmount getActual() {
        return actual;
    }

    @Nullable
    public Set<String> getSegments() {
        return segments;
    }

    public static class Builder implements DtoBuilder<DiActualQuotaUpdate> {

        @NotNull
        private final String resourceKey;
        @NotNull
        private final Set<String> segments = new HashSet<>();

        @Nullable
        private String projectKey;
        @Nullable
        private String quotaSpecKey;
        @Nullable
        private DiAmount actual;

        private Builder(@NotNull final String resourceKey) {
            if (resourceKey == null) {
                throw new IllegalArgumentException("Resource key is required");
            }
            this.resourceKey = resourceKey;
        }

        @NotNull
        public Builder project(@NotNull final String projectKey) {
            if (projectKey == null) {
                throw new IllegalArgumentException("Project key is required");
            }
            this.projectKey = projectKey;
            return this;
        }

        @NotNull
        public Builder quotaSpecKey(@NotNull final String quotaSpecKey) {
            this.quotaSpecKey = quotaSpecKey;
            return this;
        }

        @NotNull
        public Builder actual(@NotNull final DiAmount actual) {
            if (actual == null) {
                throw new IllegalArgumentException("Actual quota value is required");
            }
            this.actual = actual;
            return this;
        }

        @NotNull
        public Builder addSegments(@NotNull final Collection<String> segments) {
            if (segments.stream().anyMatch(Objects::isNull)) {
                throw new IllegalArgumentException("Segment keys may not be null");
            }
            this.segments.addAll(segments);
            return this;
        }

        @NotNull
        public Builder addSegments(@NotNull final String... segments) {
            return addSegments(Arrays.asList(segments));
        }

        @NotNull
        @Override
        public DiActualQuotaUpdate build() {
            if (projectKey == null) {
                throw new IllegalArgumentException("Project key is required");
            }
            if (actual == null) {
                throw new IllegalArgumentException("Actual quota value is required");
            }
            return new DiActualQuotaUpdate(resourceKey, projectKey, quotaSpecKey, actual, Collections.unmodifiableSet(segments));
        }

    }

    public static final class Serializer extends JsonSerializerBase<DiActualQuotaUpdate> {

        @Override
        public void serialize(@NotNull final DiActualQuotaUpdate value,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            jg.writeStartObject();
            if (value.getResourceKey() != null) {
                jg.writeStringField("resourceKey", value.getResourceKey());
            }
            if (value.getQuotaSpecKey() != null) {
                jg.writeStringField("quotaSpecKey", value.getQuotaSpecKey());
            }
            if (value.getProjectKey() != null) {
                jg.writeStringField("projectKey", value.getProjectKey());
            }
            if (value.getActual() != null) {
                jg.writeObjectFieldStart("actual");
                jg.writeNumberField("value", value.getActual().getValue());
                jg.writeObjectField("unit", value.getActual().getUnit());
                jg.writeEndObject();
            }
            if (value.getSegments() != null) {
                jg.writeObjectField("segments", value.getSegments());
            }
            jg.writeEndObject();
        }

    }

}
