package ru.yandex.qe.dispenser.api.v1.request;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.JsonSerializerBase;
import ru.yandex.qe.dispenser.api.util.NumberUtils;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.util.ValidationUtils;
import ru.yandex.qe.dispenser.api.v1.DiProject;

@JsonSerialize(using = DiEntityOwnership.Serializer.class)
@JsonDeserialize(using = DiEntityOwnership.Deserializer.class)
public final class DiEntityOwnership {
    @NotNull
    private final DiEntity entity;
    @NotNull
    private final DiProject project;
    private final int usagesCount;

    private DiEntityOwnership(@NotNull final Builder builder) {
        entity = ValidationUtils.requireNonNull(builder.entity, "Entity required!");
        project = ValidationUtils.requireNonNull(builder.project, "Project required!");
        usagesCount = NumberUtils.requirePositive(builder.usagesCount);
    }

    @NotNull
    public static Builder builder() {
        return new Builder();
    }

    @NotNull
    public DiEntity getEntity() {
        return entity;
    }

    @NotNull
    public DiProject getProject() {
        return project;
    }

    public int getUsagesCount() {
        return usagesCount;
    }

    public static final class Builder implements DtoBuilder<DiEntityOwnership> {
        @Nullable
        private DiEntity entity;
        @Nullable
        private DiProject project;
        private int usagesCount = -1;

        private Builder() {
        }

        @NotNull
        public Builder entity(@NotNull final DiEntity entity) {
            this.entity = entity;
            return this;
        }

        @NotNull
        public Builder project(@NotNull final DiProject project) {
            this.project = project;
            return this;
        }

        @NotNull
        public Builder usages(final int count) {
            usagesCount = count;
            return this;
        }

        @NotNull
        @Override
        public DiEntityOwnership build() {
            return new DiEntityOwnership(this);
        }
    }

    static final class Serializer extends JsonSerializerBase<DiEntityOwnership> {
        @Override
        public void serialize(@NotNull final DiEntityOwnership ownership,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            jg.writeStartObject();
            jg.writeObjectField("entity", ownership.getEntity());
            jg.writeObjectField("project", ownership.getProject());
            jg.writeNumberField("usagesCount", ownership.getUsagesCount());
            jg.writeEndObject();
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiEntityOwnership> {
        @NotNull
        @Override
        public DiEntityOwnership deserialize(@NotNull final JsonParser jp,
                                             @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode json = toJson(jp);
            return builder()
                    .entity(SerializationUtils.convertValue(json.get("entity"), DiEntity.class))
                    .project(SerializationUtils.convertValue(json.get("project"), DiProject.class))
                    .usages(json.get("usagesCount").asInt())
                    .build();
        }
    }
}
