package ru.yandex.qe.dispenser.api.v1.request;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.KeyBase;
import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.JsonSerializerBase;
import ru.yandex.qe.dispenser.api.util.ValidationUtils;

@JsonSerialize(using = DiEntityReference.Serializer.class)
@JsonDeserialize(using = DiEntityReference.Deserializer.class)
public class DiEntityReference extends KeyBase<String> {
    @NotNull
    private final String specificationKey;

    protected DiEntityReference(@NotNull final Builder builder) {
        super(builder);
        specificationKey = ValidationUtils.requireNonNull(builder.specificationKey, "Entity specification key is required!");
    }

    @NotNull
    public static DiEntityReference copy(@NotNull final DiEntityReference entity) {
        return withKey(entity.getKey()).bySpecification(entity.getSpecificationKey()).build();
    }

    @NotNull
    public String getSpecificationKey() {
        return specificationKey;
    }

    @NotNull
    public static Builder withKey(@NotNull final String entityKey) {
        return new Builder(entityKey);
    }

    @Override
    public boolean equals(@Nullable final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        return super.equals(o) && specificationKey.equals(((DiEntityReference) o).specificationKey);
    }

    @Override
    public int hashCode() {
        return 31 * super.hashCode() + specificationKey.hashCode();
    }

    @Override
    public String toString() {
        return "DiEntityReference{specificationKey=" + specificationKey + ", key=" + getKey() + "}";
    }

    public static class Builder extends KeyBase.Builder<String, Builder> {
        @Nullable
        public String specificationKey;

        protected Builder(@NotNull final String key) {
            super(key);
        }

        @NotNull
        public Builder bySpecification(@NotNull final String specificationKey) {
            this.specificationKey = specificationKey;
            return this;
        }

        @NotNull
        public DiEntityReference build() {
            return new DiEntityReference(this);
        }
    }

    static final class Serializer extends JsonSerializerBase<DiEntityReference> {
        @Override
        public void serialize(@NotNull final DiEntityReference entity,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            jg.writeStartObject();
            jg.writeStringField("key", entity.getKey());
            jg.writeStringField("specificationKey", entity.getSpecificationKey());
            jg.writeEndObject();
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiEntityReference> {
        @NotNull
        @Override
        public DiEntityReference deserialize(@NotNull final JsonParser jp,
                                             @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode json = toJson(jp);
            return withKey(json.get("key").asText())
                    .bySpecification(json.get("specificationKey").asText())
                    .build();
        }
    }
}
