package ru.yandex.qe.dispenser.api.v1.request;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.util.JsonSerializerBase;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.util.ValidationUtils;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiUnit;

@JsonSerialize(using = DiResourceAmount.Serializer.class)
@JsonDeserialize(using = DiResourceAmount.Deserializer.class)
public final class DiResourceAmount {
    @NotNull
    private final String resourceKey;
    @NotNull
    private final DiAmount amount;
    @NotNull
    private final Set<String> segmentKeys;

    private DiResourceAmount(@NotNull final Builder builder) {
        resourceKey = builder.resourceKey;
        amount = ValidationUtils.requireNonNull(builder.amount, "Amount is required!");
        segmentKeys = builder.segmentKeys;
    }

    @NotNull
    public static DiResourceAmount.Builder ofResource(@NotNull final String resourceKey) {
        return new DiResourceAmount.Builder(resourceKey);
    }

    @NotNull
    public String getResourceKey() {
        return resourceKey;
    }

    @NotNull
    public DiAmount getAmount() {
        return amount;
    }

    @NotNull
    public Set<String> getSegmentKeys() {
        return segmentKeys;
    }

    @NotNull
    @Override
    public String toString() {
        return resourceKey + ": " + amount;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final DiResourceAmount that = (DiResourceAmount) o;
        return resourceKey.equals(that.resourceKey) &&
                segmentKeys.equals(that.segmentKeys);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resourceKey, segmentKeys);
    }

    public static class Builder implements DtoBuilder<DiResourceAmount> {
        @NotNull
        private final String resourceKey;
        @NotNull
        private final Set<String> segmentKeys;
        @Nullable
        private DiAmount amount;

        private Builder(@NotNull final String resourceKey) {
            this.resourceKey = resourceKey;
            this.segmentKeys = new HashSet<>();
        }

        @NotNull
        public Builder withAmount(final long value, @NotNull final DiUnit unit) {
            return withAmount(DiAmount.of(value, unit));
        }

        @NotNull
        public Builder withAmount(@NotNull final DiAmount amount) {
            this.amount = amount;
            return this;
        }

        @NotNull
        public Builder withSegments(@NotNull final Collection<String> segments) {
            this.segmentKeys.addAll(segments);
            return this;
        }

        @NotNull
        public Builder withSegments(@NotNull final String... segments) {
            return withSegments(Arrays.asList(segments));
        }

        @NotNull
        @Override
        public DiResourceAmount build() {
            return new DiResourceAmount(this);
        }
    }

    static final class Serializer extends JsonSerializerBase<DiResourceAmount> {
        @Override
        public void serialize(@NotNull final DiResourceAmount amount,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            jg.writeStartObject();
            jg.writeObjectField("resourceKey", amount.getResourceKey());
            jg.writeObjectFieldStart("amount");
            jg.writeObjectField("value", amount.getAmount().getValue());
            jg.writeObjectField("unit", amount.getAmount().getUnit());
            jg.writeEndObject();
            jg.writeObjectField("segments", amount.getSegmentKeys());
            jg.writeEndObject();
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiResourceAmount> {
        @NotNull
        @Override
        public DiResourceAmount deserialize(@NotNull final JsonParser jp,
                                            @NotNull final DeserializationContext dc) throws IOException {
            final JsonNode json = toJson(jp);
            final Set<String> segments =
                    json.hasNonNull("segments") ?
                            SerializationUtils.toStringSet(json.get("segments")) :
                            Collections.emptySet();
            return ofResource(json.get("resourceKey").asText())
                    .withAmount(SerializationUtils.convertValue(json.get("amount"), DiAmount.class))
                    .withSegments(segments)
                    .build();
        }
    }
}
