package ru.yandex.qe.dispenser.api.v1.request.unbalance;

import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.DiAmount;

/**
 * DiQuotaChangeRequestUnbalancedResult.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public class DiQuotaChangeRequestUnbalancedResult {
    @NotNull
    private final String providerKey;
    private final boolean unbalanced;
    @NotNull
    private final List<DiChange> changes;

    @JsonCreator
    public DiQuotaChangeRequestUnbalancedResult(@NotNull @JsonProperty("providerKey") String providerKey,
                                                @JsonProperty("unbalanced") boolean unbalanced,
                                                @NotNull @JsonProperty("changes")  List<DiChange> changes) {
        this.providerKey = providerKey;
        this.unbalanced = unbalanced;
        this.changes = changes;
    }

    @NotNull
    public String getProviderKey() {
        return providerKey;
    }

    public boolean isUnbalanced() {
        return unbalanced;
    }

    @NotNull
    public List<DiChange> getChanges() {
        return changes;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DiQuotaChangeRequestUnbalancedResult that = (DiQuotaChangeRequestUnbalancedResult) o;
        return unbalanced == that.unbalanced && providerKey.equals(that.providerKey) && changes.equals(that.changes);
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerKey, unbalanced, changes);
    }

    @Override
    public String toString() {
        return "DiQuotaChangeRequestUnbalancedResult{" +
                "providerKey='" + providerKey + '\'' +
                ", unbalanced=" + unbalanced +
                ", changes=" + changes +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String providerKey;
        private Boolean unbalanced;
        private List<DiChange> changes;

        public Builder providerKey(String providerKey) {
            this.providerKey = providerKey;
            return this;
        }

        public Builder unbalanced(Boolean unbalanced) {
            this.unbalanced = unbalanced;
            return this;
        }

        public Builder changes(List<DiChange> changes) {
            this.changes = changes;
            return this;
        }

        public DiQuotaChangeRequestUnbalancedResult build() {
            return new DiQuotaChangeRequestUnbalancedResult(
                    Objects.requireNonNull(providerKey),
                    Objects.requireNonNull(unbalanced),
                    Objects.requireNonNull(changes)
            );
        }
    }

    public static class DiChange {
        @NotNull
        private final DiResourceKey resourceKey;
        @Nullable
        private final DiAmount recommendedMax;
        @Nullable
        private final DiAmount recommendedMin;

        @JsonCreator
        public DiChange(@NotNull @JsonProperty("resourceKey") DiResourceKey resourceKey,
                      @Nullable @JsonProperty("recommendedMax")  DiAmount recommendedMax,
                      @Nullable @JsonProperty("recommendedMin")  DiAmount recommendedMin) {
            this.resourceKey = resourceKey;
            this.recommendedMax = recommendedMax;
            this.recommendedMin = recommendedMin;
        }

        @NotNull
        public DiResourceKey getResourceKey() {
            return resourceKey;
        }

        @Nullable
        public DiAmount getRecommendedMax() {
            return recommendedMax;
        }

        @Nullable
        public DiAmount getRecommendedMin() {
            return recommendedMin;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            DiChange change = (DiChange) o;
            return resourceKey.equals(change.resourceKey) && Objects.equals(recommendedMax, change.recommendedMax)
                    && Objects.equals(recommendedMin, change.recommendedMin);
        }

        @Override
        public int hashCode() {
            return Objects.hash(resourceKey, recommendedMax, recommendedMin);
        }

        @Override
        public String toString() {
            return "DiChange{" +
                    "resourceKey=" + resourceKey +
                    ", recommendedMax=" + recommendedMax +
                    ", recommendedMin=" + recommendedMin +
                    '}';
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private DiResourceKey resourceKey;
            private DiAmount recommendedMax;
            private DiAmount recommendedMin;

            public Builder resourceKey(DiResourceKey resourceKey) {
                this.resourceKey = resourceKey;
                return this;
            }

            public Builder recommendedMax(DiAmount recommendedMax) {
                this.recommendedMax = recommendedMax;
                return this;
            }

            public Builder recommendedMin(DiAmount recommendedMin) {
                this.recommendedMin = recommendedMin;
                return this;
            }

            public DiChange build() {
                return new DiChange(
                        Objects.requireNonNull(resourceKey),
                        recommendedMax,
                        recommendedMin
                );
            }
        }
    }

    public static class DiResourceKey {
        @NotNull
        private final String resourceKey;
        private final long orderId;
        @NotNull
        private final Set<DiSegmentKey> segmentKeys;

        @JsonCreator
        public DiResourceKey(@NotNull @JsonProperty("resourceKey") String resourceKey,
                           @JsonProperty("orderId") long orderId,
                           @NotNull @JsonProperty("segmentKeys") Set<DiSegmentKey> segmentKeys) {
            this.resourceKey = resourceKey;
            this.orderId = orderId;
            this.segmentKeys = segmentKeys;
        }

        @NotNull
        public String getResourceKey() {
            return resourceKey;
        }

        public long getOrderId() {
            return orderId;
        }

        @NotNull
        public Set<DiSegmentKey> getSegmentKeys() {
            return segmentKeys;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            DiResourceKey that = (DiResourceKey) o;
            return orderId == that.orderId && Objects.equals(resourceKey, that.resourceKey)
                    && Objects.equals(segmentKeys, that.segmentKeys);
        }

        @Override
        public int hashCode() {
            return Objects.hash(resourceKey, orderId, segmentKeys);
        }

        @Override
        public String toString() {
            return "DiResourceKey{" +
                    "resourceKey='" + resourceKey + '\'' +
                    ", orderId=" + orderId +
                    ", segmentKeys=" + segmentKeys +
                    '}';
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private String resourceKey;
            private Long orderId;
            private Set<DiSegmentKey> segmentKeys;

            public Builder resourceKey(String resourceKey) {
                this.resourceKey = resourceKey;
                return this;
            }

            public Builder orderId(Long orderId) {
                this.orderId = orderId;
                return this;
            }

            public Builder segmentKeys(Set<DiSegmentKey> segmentKeys) {
                this.segmentKeys = segmentKeys;
                return this;
            }

            public DiResourceKey build() {
                return new DiResourceKey(
                        Objects.requireNonNull(resourceKey),
                        Objects.requireNonNull(orderId),
                        Objects.requireNonNull(segmentKeys)
                );
            }
        }
    }

    public static class DiSegmentKey {
        @NotNull
        private final String segmentKey;
        @NotNull
        private final String segmentationKey;

        @JsonCreator
        private DiSegmentKey(@NotNull @JsonProperty("segmentKey") String segmentKey,
                           @NotNull @JsonProperty("segmentationKey") String segmentationKey) {
            this.segmentKey = segmentKey;
            this.segmentationKey = segmentationKey;
        }

        @NotNull
        public String getSegmentKey() {
            return segmentKey;
        }

        @NotNull
        public String getSegmentationKey() {
            return segmentationKey;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            DiSegmentKey that = (DiSegmentKey) o;
            return Objects.equals(segmentKey, that.segmentKey) && Objects.equals(segmentationKey, that.segmentationKey);
        }

        @Override
        public int hashCode() {
            return Objects.hash(segmentKey, segmentationKey);
        }

        @Override
        public String toString() {
            return "DiSegmentKey{" +
                    "segmentKey='" + segmentKey + '\'' +
                    ", segmentationKey='" + segmentationKey + '\'' +
                    '}';
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private String segmentKey;
            private String segmentationKey;

            public Builder segmentKey(String segmentKey) {
                this.segmentKey = segmentKey;
                return this;
            }

            public Builder segmentationKey(String segmentationKey) {
                this.segmentationKey = segmentationKey;
                return this;
            }

            public DiSegmentKey build() {
                return new DiSegmentKey(
                        Objects.requireNonNull(segmentKey),
                        Objects.requireNonNull(segmentationKey)
                );
            }
        }
    }

}
