package ru.yandex.qe.dispenser.api.v1.response;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.util.JsonDeserializerBase;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiQuota;
import ru.yandex.qe.dispenser.api.v1.request.DiEntity;
import ru.yandex.qe.dispenser.api.v1.request.DiResourceAmount;

import static ru.yandex.qe.dispenser.api.util.SerializationUtils.convertValue;

@JsonSerialize(using = DiQuotaGetResponse.Serializer.class)
@JsonDeserialize(using = DiQuotaGetResponse.Deserializer.class)
public class DiQuotaGetResponse extends DiListResponse<DiQuota> {
    public DiQuotaGetResponse(@NotNull final Collection<DiQuota> quotas) {
        super(quotas);
    }

    @NotNull
    public static DiQuotaGetResponse of(@NotNull final DiQuota... quotas) {
        return new DiQuotaGetResponse(Arrays.asList(quotas));
    }

    public boolean canAcquire(@NotNull final String resourceKey, @NotNull final DiAmount amount, @NonNls final Set<String> segments) {
        return canAcquire(DiResourceAmount.ofResource(resourceKey).withSegments(segments).withAmount(amount).build());
    }

    public boolean canAcquire(@NotNull final String resourceKey, @NotNull final DiAmount amount) {
        return canAcquire(resourceKey, amount, Collections.emptySet());
    }

    public boolean canAcquire(@NotNull final DiResourceAmount amount) {
        final Set<DiQuota> resourceQuotas = findByResource(amount.getResourceKey(), amount.getSegmentKeys());
        if (resourceQuotas.isEmpty()) {
            throw new IllegalArgumentException("No resource '" + amount.getResourceKey() + "' in response!");
        }
        boolean result = true;
        for (final DiQuota quota : resourceQuotas) {
            result &= quota.canAcquire(amount.getAmount());
        }
        return result;
    }

    public boolean canAcquire(@NotNull final Collection<DiResourceAmount> amounts) {
        boolean result = true;
        for (final DiResourceAmount amount : amounts) {
            result &= canAcquire(amount);
        }
        return result;
    }

    public boolean canAcquire(@NotNull final DiEntity entity) {
        return canAcquire(entity.getDimensions());
    }

    public boolean canAcquireAllResources() {
        boolean result = true;
        for (final DiQuota quota : this) {
            result &= quota.canAcquire(DiAmount.anyOf(quota.getActual().getUnit()));
        }
        return result;
    }

    @NotNull
    private Set<DiQuota> findByResource(@NotNull final String resourceKey, @NotNull final Set<String> segmentKeys) {
        return stream()
                .filter(q -> q.getSpecification().getResource().getKey().equals(resourceKey) &&
                        q.getSegmentKeys().equals(segmentKeys))
                .collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public String toString() {
        return stream().map(DiQuota::toString).collect(Collectors.joining(", ", "Quotas[", "]"));
    }

    static final class Serializer extends DiListResponse.Serializer<DiQuotaGetResponse> {
        @Override
        public void serialize(@NotNull final DiQuotaGetResponse response,
                              @NotNull final JsonGenerator jg,
                              @NotNull final SerializerProvider sp) throws IOException {
            super.serialize(response, jg, sp);
        }
    }

    static final class Deserializer extends JsonDeserializerBase<DiQuotaGetResponse> {
        @NotNull
        @Override
        public DiQuotaGetResponse deserialize(@NotNull final JsonParser jp,
                                              @NotNull final DeserializationContext dc) throws IOException {
            return new DiQuotaGetResponse(convertValue(toJson(jp).get("result"), new TypeReference<Collection<DiQuota>>() {
            }));
        }
    }
}
