package ru.yandex.qe.dispenser.client.v1.impl;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.apache.cxf.jaxrs.client.WebClient;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.util.JsonSerializerBase;
import ru.yandex.qe.dispenser.api.v1.DiPerformer;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeCause;
import ru.yandex.qe.dispenser.api.v1.request.DiActionType;
import ru.yandex.qe.dispenser.api.v1.request.DiEntity;
import ru.yandex.qe.dispenser.api.v1.request.DiEntityReference;
import ru.yandex.qe.dispenser.api.v1.request.DiEntityUsage;
import ru.yandex.qe.dispenser.api.v1.request.DiOperation;
import ru.yandex.qe.dispenser.api.v1.request.DiProcessingMode;
import ru.yandex.qe.dispenser.api.v1.request.DiResourceAmount;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.api.v1.response.DiOperationResult;
import ru.yandex.qe.dispenser.api.v1.response.DiQuotaChangeResponse;
import ru.yandex.qe.dispenser.client.v1.builder.BatchQuotaChangeRequestBuilder;

final class BatchQuotaChangeRequestBuilderImpl implements BatchQuotaChangeRequestBuilder {
    @NotNull
    private final BatchChangeRequestBody body;
    @Nullable
    private String reqId;

    @NotNull
    private final Supplier<WebClient> clients;

    BatchQuotaChangeRequestBuilderImpl(@NotNull final DiProcessingMode processingMode,
                                       @NotNull final String serviceKey,
                                       @NotNull final Supplier<WebClient> clients) {
        this.clients = clients;
        this.body = new BatchChangeRequestBody(processingMode, serviceKey);
    }

    @NotNull
    @Override
    public BatchQuotaChangeRequestBuilder acquireResource(@NotNull final DiResourceAmount amount,
                                                          @NotNull final DiPerformer performer,
                                                          @NotNull final DiQuotaChangeCause cause) {
        return addOperation(new DiOperation<>(DiActionType.FORCE_ACQUIRE_RESOURCE, amount, performer, cause));
    }

    @NotNull
    @Override
    public BatchQuotaChangeRequestBuilder releaseResource(@NotNull final DiResourceAmount amount,
                                                          @NotNull final DiPerformer performer,
                                                          @NotNull final DiQuotaChangeCause cause) {
        return addOperation(new DiOperation<>(DiActionType.RELEASE_RESOURCE, amount, performer, cause));
    }

    @NotNull
    @Override
    public BatchQuotaChangeRequestBuilder createEntity(@NotNull final DiEntity entity,
                                                       @NotNull final DiPerformer performer,
                                                       @NotNull final DiQuotaChangeCause cause) {
        return addOperation(new DiOperation<>(DiActionType.CREATE_ENTITY, entity, performer, cause));
    }

    @NotNull
    @Override
    public BatchQuotaChangeRequestBuilder releaseEntity(@NotNull final DiEntityReference entity,
                                                        @NotNull final DiQuotaChangeCause cause) {
        return addOperation(new DiOperation<>(DiActionType.RELEASE_ENTITY, DiEntityReference.copy(entity), null, cause));
    }

    @NotNull
    @Override
    public BatchQuotaChangeRequestBuilder shareEntity(@NotNull final DiEntityUsage entityUsage,
                                                      @NotNull final DiPerformer performer,
                                                      @NotNull final DiQuotaChangeCause cause) {
        return addOperation(new DiOperation<>(DiActionType.SHARE_ENTITY, entityUsage, performer, cause));
    }

    @NotNull
    @Override
    public BatchQuotaChangeRequestBuilder releaseEntitySharing(@NotNull final DiEntityUsage entityUsage,
                                                               @NotNull final DiPerformer performer,
                                                               @NotNull final DiQuotaChangeCause cause) {
        return addOperation(new DiOperation<>(DiActionType.RELEASE_ENTITY_SHARING, entityUsage, performer, cause));
    }

    @NotNull
    @Override
    public BatchQuotaChangeRequestBuilder releaseAllEntitySharings(@NotNull final DiEntityReference entity,
                                                                   @NotNull final DiPerformer performer,
                                                                   @NotNull final DiQuotaChangeCause cause) {
        return addOperation(new DiOperation<>(DiActionType.RELEASE_ALL_ENTITY_SHARINGS, DiEntityReference.copy(entity), performer, cause));
    }

    @NotNull
    private BatchQuotaChangeRequestBuilder addOperation(@NotNull final DiOperation<?> operation) {
        body.add(operation);
        return this;
    }

    @NotNull
    @Override
    public BatchQuotaChangeRequestBuilder withReqId(@NotNull final String reqId) {
        this.reqId = reqId;
        return this;
    }

    @NotNull
    @Override
    public DiQuotaChangeResponse perform() {
        final BatchChangeResponseBody response = clients.get()
                .path("/change-quotas")
                .query("reqId", WebClientUtils.actualReqId(reqId))
                .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                .post(body, BatchChangeResponseBody.class);
        final List<DiOperationResult> operationResults = response.stream()
                .map(r -> new DiOperationResult(body.operations.get(r.operationId), r.isSuccess, r.errorReason))
                .collect(Collectors.toList());
        return new DiQuotaChangeResponse(operationResults);
    }

    @JsonSerialize(using = BatchChangeRequestBody.Serializer.class)
    static final class BatchChangeRequestBody {
        @NotNull
        private final DiProcessingMode mode;
        @NotNull
        private final String serviceKey;
        @NotNull
        private final Map<Integer, DiOperation<?>> operations = new HashMap<>();

        @NotNull
        private final AtomicInteger operationIdGenerator = new AtomicInteger(0);

        BatchChangeRequestBody(@NotNull final DiProcessingMode mode, @NotNull final String serviceKey) {
            this.mode = mode;
            this.serviceKey = serviceKey;
        }

        void add(@NotNull final DiOperation<?> operation) {
            operations.put(operationIdGenerator.incrementAndGet(), operation);
        }

        int size() {
            return operations.size();
        }

        @Override
        public boolean equals(@Nullable final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final BatchChangeRequestBody that = (BatchChangeRequestBody) o;
            return mode == that.mode && serviceKey.equals(that.serviceKey) && operations.equals(that.operations);
        }

        @Override
        public int hashCode() {
            return 31 * (31 * mode.hashCode() + serviceKey.hashCode()) + operations.hashCode();
        }

        static class Serializer extends JsonSerializerBase<BatchChangeRequestBody> {
            @Override
            public void serialize(@NotNull final BatchChangeRequestBody value,
                                  @NotNull final JsonGenerator jg,
                                  @NotNull final SerializerProvider sp) throws IOException {
                jg.writeStartObject();
                jg.writeObjectField("mode", value.mode);
                jg.writeObjectField("serviceKey", value.serviceKey);
                jg.writeArrayFieldStart("operations");
                for (final Map.Entry<Integer, DiOperation<?>> operation : value.operations.entrySet()) {
                    jg.writeStartObject();
                    jg.writeObjectField("id", operation.getKey());
                    jg.writeObjectField("operation", operation.getValue());
                    jg.writeEndObject();
                }
                jg.writeEndArray();
                jg.writeEndObject();
            }
        }
    }

    static final class BatchChangeResponseBody extends DiListResponse<BatchChangeResponseBody.OperationResultReference> {
        BatchChangeResponseBody(@JsonProperty("result") final Collection<OperationResultReference> result) {
            super(result);
        }

        static final class OperationResultReference {
            private final int operationId;
            private final boolean isSuccess;
            @Nullable
            private final String errorReason;

            OperationResultReference(@JsonProperty("operationId") final int operationId,
                                     @JsonProperty("isSuccess") final boolean isSuccess,
                                     @JsonProperty("errorReason") @Nullable final String errorReason) {
                this.operationId = operationId;
                this.isSuccess = isSuccess;
                this.errorReason = errorReason;
            }
        }
    }

    @JsonSerialize(using = EntityUsageAction.Serializer.class)
    private final class EntityUsageAction {
        @NotNull
        private final DiActionType type;
        @NotNull
        private final DiEntityUsage usage;

        private EntityUsageAction(@NotNull final DiActionType type, @NotNull final DiEntityUsage usage) {
            this.type = type;
            this.usage = usage;
        }

        class Serializer extends JsonSerializerBase<EntityUsageAction> {
            @Override
            public void serialize(@NotNull final EntityUsageAction action,
                                  @NotNull final JsonGenerator jg,
                                  @NotNull final SerializerProvider sp) throws IOException {
                jg.writeStartObject();
                jg.writeObjectField("type", action.type);
                jg.writeObjectFieldStart("entity");
                jg.writeObjectField("key", action.usage.getEntity().getKey());
                jg.writeObjectField("specificationKey", action.usage.getEntity().getSpecificationKey());
                jg.writeEndObject();
                jg.writeObjectField("usagesCount", action.usage.getUsagesCount());
                jg.writeEndObject();
            }
        }
    }
}
