package ru.yandex.qe.dispenser.client.v1.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.ws.rs.HttpMethod;

import org.apache.cxf.common.util.UrlUtils;
import org.apache.cxf.helpers.IOUtils;
import org.apache.cxf.message.Message;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DiRequest;

public final class DiRequestImpl<T> implements DiRequest<T> {
    @NotNull
    private final Message m;

    public DiRequestImpl(@NotNull final Message message) {
        this.m = message;
    }

    @NotNull
    @Override
    public String getMethod() {
        return Optional.ofNullable((String) m.get(Message.HTTP_REQUEST_METHOD)).orElse(HttpMethod.GET);
    }

    @Nullable
    @Override
    public String getFirstHeader(@NotNull final String name) {
        return getHeaders().getOrDefault(name, Collections.emptyList()).stream().findFirst().orElse(null);
    }

    @NotNull
    @Override
    @SuppressWarnings("unchecked")
    public Map<String, List<String>> getHeaders() {
        return (Map<String, List<String>>) Optional.ofNullable(m.get(Message.PROTOCOL_HEADERS)).orElse(Collections.emptyMap());
    }

    @Override
    public void setHeader(@NotNull final String name, @NotNull final String value) {
        getHeaders().put(name, Collections.singletonList(value));
    }

    @NotNull
    public String getRequestUrl() {
        final String requestUrl = (String) m.get(Message.REQUEST_URL);
        if (requestUrl != null) {
            return requestUrl;
        }
        final String requestUri = (String) m.get(Message.REQUEST_URI);
        if (requestUri != null) {
            return requestUri;
        }
        return "";
    }

    @NotNull
    @Override
    public String getFullUrl() {
        final String requestUrl = getRequestUrlParts()[0];
        final String queryString = getQueryString();
        return !queryString.isEmpty() ? requestUrl + '?' + queryString : requestUrl;
    }

    @NotNull
    @Override
    public String getPath() {
        final String pathInfo = (String) m.get(Message.PATH_INFO);
        return pathInfo != null ? pathInfo : getRequestUrlParts()[0];
    }

    @NotNull
    public String getQueryString() {
        final String queryString = (String) m.get(Message.QUERY_STRING);
        return queryString != null ? queryString : getRequestUrlParts()[1];
    }

    @NotNull
    private String[] getRequestUrlParts() {
        final String[] parts = getRequestUrl().split("\\?", 2);
        if (parts.length == 0) {
            return new String[]{"", ""};
        }
        if (parts.length == 1) {
            return new String[]{parts[0], ""};
        }
        return parts;
    }

    @Nullable
    @Override
    public String getQueryParam(@NotNull final String name) {
        return UrlUtils.parseQueryString(getQueryString()).get(name.toLowerCase());
    }

    @Nullable
    @Override
    public T getBody() {
        final List<?> content = m.getContent(List.class);
        return !content.isEmpty() ? (T) content.get(0) : null;
    }

    @Override
    public String toString() {
        String data = null;
        if (m.getContentFormats().contains(InputStream.class)) {
            final byte[] body;
            try {
                body = IOUtils.readBytesFromStream(m.getContent(InputStream.class));
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            data = new String(body, StandardCharsets.UTF_8);
            m.setContent(InputStream.class, new ByteArrayInputStream(body));
        }

        final StringBuilder sb = new StringBuilder("curl -X ").append(getMethod()).append(" '").append(getFullUrl()).append("' -i ");
        getHeaders().forEach((name, values) -> {
            values.forEach(value -> sb.append("-H '").append(name).append(":").append(isPrivateHeader(name) ? "***" : value).append("' "));
        });
        if (data != null) {
            sb.append(" -d '").append(data).append("' ");
        }
        sb.append("| less");
        return sb.toString();
    }

    private static final String[] PRIVATE_HEADERS = {
            "Authorization",
            "Cookie"
    };

    public static boolean isPrivateHeader(@NotNull final String headerName) {
        return Arrays.stream(PRIVATE_HEADERS).anyMatch(headerName::equalsIgnoreCase);
    }
}
