package ru.yandex.qe.dispenser.client.v1.impl;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Objects;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.dispenser.api.util.TextUtils;
import ru.yandex.qe.dispenser.client.v1.DiOAuthToken;

public final class DispenserConfig {
    private static final Logger LOG = LoggerFactory.getLogger(DispenserConfig.class);

    @Nullable
    private String dispenserHost;
    @Nullable
    private DiOAuthToken token;

    private long connectionTimeout = 3000;
    private long receiveTimeout = 6000;
    private boolean allowRedirects = false;

    @NotNull
    private final ClientInfo info = new ClientInfo();

    public DispenserConfig() {
        final Package configPackage = getClass().getPackage();
        info.groupId = configPackage.getImplementationVendor();
        info.artifactId = configPackage.getImplementationTitle();
        info.version = configPackage.getImplementationVersion();
        try {
            info.hostName = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            LOG.error("Can't resolve localhost!", e);
        }
    }

    @NotNull
    public DispenserConfig setClientId(@NotNull final String clientId) {
        TextUtils.requireUpperCase(clientId, "Client ID must be not empty upper case -- " + clientId);
        this.info.clientId = clientId;
        return this;
    }

    @NotNull
    public DispenserConfig setEnvironment(@NotNull final Environment environment) {
        this.info.environment = environment;
        return this;
    }

    @NotNull
    public ClientInfo getInfo() {
        return info;
    }

    @NotNull
    public String getDispenserHost() {
        return Objects.requireNonNull(dispenserHost, "Dispenser host is required!");
    }


    @NotNull
    public DispenserConfig setDispenserHost(@NotNull final String host) {
        this.dispenserHost = host;
        if (info.environment == null) {
            info.environment = Environment.byDispenserHost(host);
        }
        return this;
    }

    @NotNull
    public DiOAuthToken getToken() {
        return Objects.requireNonNull(token, "Your zombie OAuthToken is required!");
    }

    @NotNull
    public DispenserConfig setServiceZombieOAuthToken(@NotNull final DiOAuthToken token) {
        this.token = token;
        return this;
    }

    long getConnectionTimeout() {
        return connectionTimeout;
    }

    @NotNull
    public DispenserConfig setConnectionTimeout(final long connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
        return this;
    }

    long getReceiveTimeout() {
        return receiveTimeout;
    }

    @NotNull
    public DispenserConfig setReceiveTimeout(final long receiveTimeout) {
        this.receiveTimeout = receiveTimeout;
        return this;
    }

    boolean isAllowRedirects() {
        return allowRedirects;
    }

    @NotNull
    public DispenserConfig setAllowRedirects(final boolean allowRedirects) {
        this.allowRedirects = allowRedirects;
        return this;
    }

    public enum Environment {
        DEVELOPMENT("dispenser-dev.yandex-team.ru"),
        LOAD_TESTING("dispenser-load.qloud.yandex-team.ru"),
        TESTING("dispenser-test.yandex-team.ru"),
        PRODUCTION("dispenser.yandex-team.ru");

        @NotNull
        private final String dispenserHost;

        Environment(@NotNull final String dispenserHost) {
            this.dispenserHost = dispenserHost;
        }

        @NotNull
        public String getDispenserHost() {
            return dispenserHost;
        }

        @Nullable
        public static Environment byDispenserHost(@NotNull final String host) {
            for (final Environment env : values()) {
                if (host.endsWith(env.dispenserHost)) {
                    return env;
                }
            }
            return null;
        }
    }

    public static final class ClientInfo {
        @Nullable
        private String clientId;
        @Nullable
        private Environment environment;
        @Nullable
        private String hostName;
        @Nullable
        private String artifactId;
        @Nullable
        private String groupId;
        @Nullable
        private String version;

        @NotNull
        public String getClientId() {
            return Objects.requireNonNull(clientId, "Client ID is required!");
        }

        @NotNull
        public Environment getEnvironment() {
            return Objects.requireNonNull(environment, "Environment is required!");
        }

        @Nullable
        public String getHostName() {
            return hostName;
        }

        @Nullable
        public String getArtifactId() {
            return artifactId;
        }

        @Nullable
        public String getGroupId() {
            return groupId;
        }

        @Nullable
        public String getVersion() {
            return version;
        }
    }
}
