package ru.yandex.qe.dispenser.client.v1.impl;

import java.util.Collections;
import java.util.List;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.jaxrs.json.JacksonJaxbJsonProvider;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import org.apache.cxf.jaxrs.client.ClientConfiguration;
import org.apache.cxf.jaxrs.client.JAXRSClientFactoryBean;
import org.apache.cxf.jaxrs.client.WebClient;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.util.MoreHeaders;
import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.client.v1.Dispenser;
import ru.yandex.qe.dispenser.client.v1.DispenserFactory;

abstract class DispenserFactoryImpl implements DispenserFactory {

    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper().registerModules(new Jdk8Module(),
            new JavaTimeModule(), new KotlinModule.Builder().build());

    @NotNull
    protected final DispenserConfig config;

    protected DispenserFactoryImpl(@NotNull final DispenserConfig config) {
        this.config = config;
    }

    @NotNull
    abstract WebClient createUnconfiguredClient(@NotNull final String address);

    @NotNull
    List<?> getProviders() {
        return Collections.singletonList(new JacksonJaxbJsonProvider(OBJECT_MAPPER, JacksonJaxbJsonProvider.DEFAULT_ANNOTATIONS));
    }

    @NotNull
    WebClient createConfiguredWebClient() {
        final JAXRSClientFactoryBean bean = new CustomJAXRSClientFactoryBean(this::createUnconfiguredClient);
        bean.setAddress(config.getDispenserHost() + "/api");
        bean.setProviders(getProviders());
        final WebClient client = bean.createWebClient();

        final ClientConfiguration clientConfig = WebClient.getConfig(client);
        clientConfig.getOutInterceptors().add(new UriLogger());
        clientConfig.getInInterceptors().add(new ErrorMessageLogger());

        client.header(MoreHeaders.X_DISPENSER_CLIENT_INFO, SerializationUtils.writeValueAsString(config.getInfo()));

        return client;
    }

    @NotNull
    @Override
    public Dispenser get() {
        return new DispenserImpl(this::createConfiguredWebClient, config.getToken());
    }
}
