package ru.yandex.qe.dispenser.client.v1.impl;

import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.function.Supplier;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;
import javax.ws.rs.HttpMethod;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;

import org.apache.cxf.configuration.jsse.TLSClientParameters;
import org.apache.cxf.jaxrs.client.WebClient;
import org.apache.cxf.transport.http.HTTPConduit;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;

import ru.yandex.qe.dispenser.api.v1.DiBotSyncStatus;
import ru.yandex.qe.dispenser.api.v1.DiCheck;
import ru.yandex.qe.dispenser.api.v1.DiDispenserSettings;
import ru.yandex.qe.dispenser.api.v1.DiEntitySpec;
import ru.yandex.qe.dispenser.api.v1.DiPermission;
import ru.yandex.qe.dispenser.api.v1.DiPersonInfo;
import ru.yandex.qe.dispenser.api.v1.DiProject;
import ru.yandex.qe.dispenser.api.v1.DiQuotaLightView;
import ru.yandex.qe.dispenser.api.v1.DiQuotaMaxDeltaUpdate;
import ru.yandex.qe.dispenser.api.v1.DiQuotaMaxUpdate;
import ru.yandex.qe.dispenser.api.v1.DiResource;
import ru.yandex.qe.dispenser.api.v1.DiResourceGroup;
import ru.yandex.qe.dispenser.api.v1.DiSegment;
import ru.yandex.qe.dispenser.api.v1.DiSegmentation;
import ru.yandex.qe.dispenser.api.v1.DiService;
import ru.yandex.qe.dispenser.api.v1.field.DiField;
import ru.yandex.qe.dispenser.api.v1.project.DiExtendedProject;
import ru.yandex.qe.dispenser.api.v1.request.DiProcessingMode;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.api.v1.response.DiQuotaGetResponse;
import ru.yandex.qe.dispenser.client.QueryableRequestBuilder;
import ru.yandex.qe.dispenser.client.RequestBuilder;
import ru.yandex.qe.dispenser.client.v1.DiOAuthToken;
import ru.yandex.qe.dispenser.client.v1.DiPerson;
import ru.yandex.qe.dispenser.client.v1.Dispenser;
import ru.yandex.qe.dispenser.client.v1.builder.BatchQuotaChangeRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.BatchWeakQuotaChangeRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.CreateRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.DeleteRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.DispenserSettingsModificationRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.GetEntitiesRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.GetEntityOwnershipsRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.GetEntitySpecsRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.GetProjectMetaRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.GetProjectsRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.GetQuotasRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.GetWeakQuotasRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.PersonGroupModificationRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.ProjectModificationRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.QuotaMaxUpdateRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.ResourceModificationRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.ServiceModificationRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.SyncActualQuotasRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.SyncQuotasRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.SyncRawQuotasRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.UpdateEntitiesRequestBuilder;
import ru.yandex.qe.http.certificates.TrustManagersProvider;

@SuppressWarnings("InnerClassTooDeeplyNested")
class DispenserImpl implements Dispenser {
    private static final String UPDATE_OPERATIONS_MESSAGE = "Update operations are not implemented yet";

    @NotNull
    private final Supplier<WebClient> clients;

    DispenserImpl(@NotNull final Supplier<WebClient> clients, final @NotNull DiOAuthToken token) {
        this.clients = () -> {
            final WebClient webClient = clients.get()
                    .path("/v1")
                    .header(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.AUTHORIZATION, "OAuth " + token.getValue());

            final HTTPConduit conduit = WebClient.getConfig(webClient)
                    .getHttpConduit();
            //HTTPConduit is not available in spy mode
            //there is no need to use yandex CA in spy mode
            if (conduit != null) {
                TLSClientParameters params =
                        conduit.getTlsClientParameters();

                if (params == null) {
                    params = new TLSClientParameters();
                    conduit.setTlsClientParameters(params);
                }
                params.setTrustManagers(TrustManagersProvider.getInstance().getTrustManagers());
                params.setHostnameVerifier(new HostnameVerifier() {
                    @Override
                    public boolean verify(final String s, final SSLSession sslSession) {
                        return s.equals(sslSession.getPeerHost());
                    }
                });
            }
            return webClient;
        };
    }

    @Override
    public @NotNull DispenserSettingsApi dispenserSettings() {
        return new DispenserSettingsApiImpl();
    }

    @NotNull
    @Override
    public Dispenser.ProjectApi projects() {
        return new ProjectApiImpl();
    }

    @NotNull
    @Override
    public Dispenser.CrudProjectApi project(@NotNull final String projectKey) {
        return new CrudProjectApiImpl(projectKey);
    }

    @NotNull
    @Override
    public Dispenser.ServiceApi service(@NotNull final String serviceKey) {
        return new ServiceApiImpl(serviceKey);
    }

    @NotNull
    @Override
    public Dispenser.QuotaApi quotas() {
        return new QuotaApiImpl();
    }

    @NotNull
    @Override
    public Dispenser.EntitySpecApi entitySpecifications() {
        return new EntitySpecApiImpl();
    }

    @NotNull
    @Override
    public GetEntitiesRequestBuilder<?> getEntities() {
        return new GetEntitiesRequestBuilderImpl(clients);
    }

    @NotNull
    @Override
    public UpdateEntitiesRequestBuilder updateEntities() {
        return new UpdateEntitiesRequestBuilderImpl(clients);
    }

    @NotNull
    @Override
    public GetEntityOwnershipsRequestBuilder<?> getEntityOwnerships() {
        return new GetEntityOwnershipsRequestBuilderImpl(clients);
    }

    @NotNull
    @Override
    public Dispenser.WeakQuotaApi weakQuotas() {
        return new WeakQuotaApiImpl();
    }

    @NotNull
    @Override
    public Dispenser.PersonApi persons() {
        return new PersonApiImpl();
    }

    @NotNull
    @Override
    public Dispenser.PermissionApi permission() {
        return new PermissionApiImpl();
    }

    @NotNull
    @Override
    public Dispenser.SegmentationApi segmentations() {
        return new SegmentationApiImpl();
    }

    @NotNull
    @Override
    public Dispenser.QuotaChangeRequestApi quotaChangeRequests() {
        return new QuotaChangeRequestApiImpl(clients);
    }

    @Override
    @NotNull
    public Dispenser.PropertyApi properties() {
        return new PropertyApiImpl(clients);
    }

    private final class ProjectApiImpl implements ProjectApi {
        @NotNull
        @Override
        @TestOnly
        public CreateRequestBuilder<DiProject> create(final DiProject project) {
            return new CreateRequestBuilder<DiProject>() {
                @Nullable
                private String reqId;

                @NotNull
                @Override
                public DiProject performBy(@NotNull final DiPerson person) {
                    final String createSubprojectPath = new StringJoiner("/")
                            .add("projects")
                            .add(Objects.requireNonNull(project.getParentProjectKey(), "Can't create project without parent"))
                            .add("create-subproject")
                            .toString();
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path(createSubprojectPath)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .post(project, DiProject.class);
                }

                @Override
                public @NotNull CreateRequestBuilder<DiProject> withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @NotNull
        @Override
        @TestOnly
        public CreateRequestBuilder<DiProject> update(final DiProject project) {
            return new CreateRequestBuilder<DiProject>() {
                @Nullable
                private String reqId;

                @NotNull
                @Override
                public DiProject performBy(@NotNull final DiPerson person) {
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path("/projects/" + project.getKey())
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .post(project, DiProject.class);
                }

                @Override
                public @NotNull CreateRequestBuilder<DiProject> withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @NotNull
        @Override
        public GetProjectsRequestBuilder<DiProject> get() {
            return new GetProjectsRequestBuilderImpl(clients);
        }

        @NotNull
        @Override
        public GetProjectsRequestBuilder<DiExtendedProject> getWithFields(@NotNull final DiField<?>... fields) {
            return new GetProjectsFieldsRequestBuilderImpl(clients, fields);
        }

        @NotNull
        @Override
        public GetProjectMetaRequestBuilder<?> getMeta() {
            return new GetProjectMetaRequestBuilderImpl(clients);
        }
    }

    // TODO Replace with new CrudApi
    private final class CrudProjectApiImpl implements CrudProjectApi {
        private static final String MEMBERS = "members";
        private static final String RESPONSIBLES = "responsibles";

        private final String projectKey;
        private final String path;

        CrudProjectApiImpl(@NotNull final String projectKey) {
            this.projectKey = projectKey;
            this.path = "projects/" + projectKey;
        }

        @NotNull
        @TestOnly
        @Override
        public ProjectModificationRequestBuilder create() {
            return new ProjectModificationRequestBuilder(projectKey) {
                private String reqId = null;

                @NotNull
                @Override
                public DiProject performBy(final @NotNull DiPerson person) {
                    final String createSubprojectPath = new StringJoiner("/")
                            .add("projects")
                            .add(Objects.requireNonNull(parentProjectKey, "Can't create project without parent"))
                            .add("create-subproject")
                            .toString();
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path(createSubprojectPath)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .post(buildProject(), DiProject.class);
                }

                @NotNull
                @Override
                public ProjectModificationRequestBuilder withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @Override
        @NotNull
        public RequestBuilder<DiProject> get() {
            return () -> clients.get().path(path).get(DiProject.class);
        }

        @NotNull
        @TestOnly
        @Override
        public ProjectModificationRequestBuilder update() {
            return new ProjectModificationRequestBuilder(projectKey) {
                private String reqId;

                @Override
                public @NotNull DiProject performBy(final @NotNull DiPerson person) {
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path("/projects/" + projectKey)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .post(buildProject(), DiProject.class);
                }

                @NotNull
                @Override
                public ProjectModificationRequestBuilder withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @Override
        public DeleteRequestBuilder<DiProject> delete() {
            return new DeleteRequestBuilderImpl<>(clients, path, DiProject.class);
        }

        @Override
        public @NotNull PersonGroupModificationRequestBuilder<DiProject> members() {
            return new PersonGroupModificationRequestBuilderImpl<>(clients, path, MEMBERS, DiProject.class);
        }

        @Override
        public @NotNull PersonGroupModificationRequestBuilder<DiProject> responsibles() {
            return new PersonGroupModificationRequestBuilderImpl<>(clients, path, RESPONSIBLES, DiProject.class);
        }

        @Override
        public ItemsCrudApi<DiCheck, DiCheck.Body, ItemCrudApi<DiCheck, DiCheck.Body>> checks() {
            return new ChecksApiImpl(path, clients);
        }
    }

    private final class DispenserSettingsApiImpl implements Dispenser.DispenserSettingsApi {
        private static final String ADMINS = "admins";

        private final String path;

        private DispenserSettingsApiImpl() {
            this.path = "settings/";
        }

        @NotNull
        @TestOnly
        @Override
        public DispenserSettingsModificationRequestBuilder create() {
            throw new RuntimeException("DispenserSettings object can't be created!");
        }

        @NotNull
        @Override
        public RequestBuilder<DiDispenserSettings> get() {
            return () -> clients.get().path(path).get(DiDispenserSettings.class);
        }

        @NotNull
        @TestOnly
        @Override
        public DispenserSettingsModificationRequestBuilder update() {
            return new DispenserSettingsModificationRequestBuilder() {
                private String reqId;

                @Override
                public @NotNull DiDispenserSettings performBy(final @NotNull DiPerson person) {
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path(path)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .invoke(HttpMethod.PATCH, buildDispenser(), DiDispenserSettings.class);
                }

                @NotNull
                @Override
                public DispenserSettingsModificationRequestBuilder withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @Override
        public DeleteRequestBuilder<DiDispenserSettings> delete() {
            throw new RuntimeException("DispenserSettings object can't be deleted by anyone!");
        }

    }


    private final class ServiceApiImpl implements Dispenser.ServiceApi {
        private static final String ADMINS = "admins";
        private static final String TRUSTEES = "trustees";

        private final String serviceKey;
        private final String path;

        private ServiceApiImpl(final String serviceKey) {
            this.serviceKey = serviceKey;
            this.path = "services/" + serviceKey;
        }

        @NotNull
        @TestOnly
        @Override
        public ServiceModificationRequestBuilder create() {
            return new ServiceModificationRequestBuilder(serviceKey) {
                private String reqId;

                @Override
                public @NotNull DiService performBy(final @NotNull DiPerson person) {
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path(path)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .put(buildService(), DiService.class);
                }

                @NotNull
                @Override
                public ServiceModificationRequestBuilder withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @NotNull
        @Override
        public RequestBuilder<DiService> get() {
            return () -> clients.get().path(path).get(DiService.class);
        }

        @NotNull
        @TestOnly
        @Override
        public ServiceModificationRequestBuilder update() {
            return new ServiceModificationRequestBuilder(serviceKey) {
                private String reqId;

                @Override
                public @NotNull DiService performBy(final @NotNull DiPerson person) {
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path(path)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .post(buildService(), DiService.class);
                }

                @NotNull
                @Override
                public ServiceModificationRequestBuilder withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @Override
        public DeleteRequestBuilder<DiService> delete() {
            return new DeleteRequestBuilderImpl<>(clients, path, DiService.class);
        }

        @Override
        public @NotNull PersonGroupModificationRequestBuilder<DiService> admins() {
            return new PersonGroupModificationRequestBuilderImpl<>(clients, path, ADMINS, DiService.class);
        }

        @Override
        public @NotNull PersonGroupModificationRequestBuilder<DiService> trustees() {
            return new PersonGroupModificationRequestBuilderImpl<>(clients, path, TRUSTEES, DiService.class);
        }

        @NotNull
        @Override
        public Dispenser.ServiceResourceApi resource(@NotNull final String resourceKey) {
            return new ServiceResourceApiImpl(resourceKey, path);
        }

        @NotNull
        @Override
        public Dispenser.SyncStateApi syncState() {
            return new SyncStateApiImpl(serviceKey);
        }

        @Override
        public @NotNull UpdateMaxApi updateMax() {
            return new UpdateMaxApiImpl(serviceKey);
        }

        @Override
        public ResourceGroupApi resourceGroups() {
            return new ResourceGroupApiImpl(clients, path);
        }

        @Override
        public @NotNull RequestBuilder<DiListResponse<DiResource>> resources() {
            return () -> clients.get().path(path + "/resources").get(new GenericType<DiListResponse<DiResource>>() {
            });
        }

        @Override
        public @NotNull RequestBuilder<DiListResponse<DiResource>> resourcesByGroups(@NotNull final String... groups) {
            return () -> clients.get().path(path + "/resources").query("group", (Object[]) groups).get(new GenericType<DiListResponse<DiResource>>() {
            });
        }
    }

    private final class ServiceResourceApiImpl implements Dispenser.ServiceResourceApi {
        @NotNull
        private final String resourceKey;
        @NotNull
        private final String path;

        ServiceResourceApiImpl(@NotNull final String resourceKey, @NotNull final String basePath) {
            this.resourceKey = resourceKey;
            this.path = basePath + "/resources/" + resourceKey;
        }

        @NotNull
        @TestOnly
        @Override
        public ResourceModificationRequestBuilder create() {
            return new ResourceModificationRequestBuilder(resourceKey) {
                private String reqId;

                @Override
                public @NotNull DiResource performBy(@NotNull final DiPerson person) {
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path(path)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .put(buildResource(), DiResource.class);
                }

                @NotNull
                @Override
                public ResourceModificationRequestBuilder withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @NotNull
        @Override
        public RequestBuilder<DiResource> get() {
            return () -> clients.get().path(path).get(DiResource.class);
        }

        @NotNull
        @TestOnly
        @Override
        public ResourceModificationRequestBuilder update() {
            return new ResourceModificationRequestBuilder(resourceKey) {
                private String reqId;

                @Override
                public @NotNull DiResource performBy(@NotNull final DiPerson person) {
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path(path)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .post(buildResource(), DiResource.class);
                }

                @NotNull
                @Override
                public ResourceModificationRequestBuilder withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }

        @Override
        public DeleteRequestBuilder<DiResource> delete() {
            return new DeleteRequestBuilderImpl<>(clients, path, DiResource.class);
        }

        @Override
        public ResourceSegmentaionApi segmentations() {
            return new ResourseSegmentationApiImpl(clients, path);
        }
    }

    private final class WeakQuotaApiImpl implements WeakQuotaApi {
        @NotNull
        @Override
        public GetWeakQuotasRequestBuilder<?> get() {
            return new GetWeakQuotasRequestBuilderImpl(clients);
        }

        @NotNull
        @Override
        public BatchWeakQuotaChangeRequestBuilder changeInService(final @NotNull String serviceKey,
                                                                  final @NotNull String entitySpecId) {
            return new BatchWeakQuotaChangeRequestBuilderImpl(serviceKey, entitySpecId, clients);
        }
    }

    private final class QuotaApiImpl implements QuotaApi {
        @Override
        public @NotNull GetQuotasRequestBuilder<DiQuotaGetResponse, ?> get() {
            return new GetFullQuotasRequestBuilderImpl(clients);
        }

        @Override
        public @NotNull GetQuotasRequestBuilder<DiListResponse<DiQuotaLightView>, ?> getLightViews() {
            return new GetQuotaLightViewsRequestBuilderImpl(clients);
        }

        @NotNull
        @Override
        public BatchQuotaChangeRequestBuilder changeInService(@NotNull final String serviceKey,
                                                              @NotNull final DiProcessingMode processingMode) {
            return new BatchQuotaChangeRequestBuilderImpl(processingMode, serviceKey, clients);
        }
    }

    private final class EntitySpecApiImpl implements EntitySpecApi {
        @Override
        public @NotNull GetEntitySpecsRequestBuilder get() {
            return new GetEntitySpecsRequestBuilderImpl(clients);
        }

        @NotNull
        @Override
        public RequestBuilder<DiEntitySpec> get(@NotNull final String serviceKey, @NotNull final String entitySpecKey) {
            return () -> clients.get().path("/entity-specifications/" + serviceKey + "/" + entitySpecKey).get(DiEntitySpec.class);
        }
    }

    private final class SyncStateApiImpl implements Dispenser.SyncStateApi {
        @NotNull
        private final String serviceKey;

        private SyncStateApiImpl(@NotNull final String serviceKey) {
            this.serviceKey = serviceKey;
        }

        @NotNull
        @Override
        public SyncQuotasRequestBuilder quotas() {
            return new SyncQuotasRequestBuilderImpl(serviceKey, clients);
        }

        @NotNull
        @Override
        public SyncActualQuotasRequestBuilder actualQuotas() {
            return new SyncActualQuotasRequestBuilderImpl(serviceKey, clients);
        }

        @NotNull
        @Override
        public SyncRawQuotasRequestBuilder rawQuotas() {
            return new SyncRawQuotasRequestBuilderImpl(serviceKey, clients);
        }

    }

    private final class UpdateMaxApiImpl implements Dispenser.UpdateMaxApi {
        @NotNull
        private final String serviceKey;

        private UpdateMaxApiImpl(@NotNull final String serviceKey) {
            this.serviceKey = serviceKey;
        }

        @Override
        public @NotNull QuotaMaxUpdateRequestBuilder<DiQuotaMaxUpdate> quotas() {
            return new QuotaMaxUpdateRequestBuilderImpl<>(serviceKey, clients, "quotas");
        }

        @Override
        public @NotNull QuotaMaxUpdateRequestBuilder<DiQuotaMaxDeltaUpdate> deltas() {
            return new QuotaMaxUpdateRequestBuilderImpl<>(serviceKey, clients, "deltas");
        }
    }

    private final class PersonApiImpl implements PersonApi {
        @Override
        public QueryableRequestBuilder<DiPersonInfo> getInfo() {
            return new QueryableRequestBuilderImpl<>(new GenericType<>(DiPersonInfo.class), () -> clients.get().path("/persons/info"));
        }
    }

    private final class PermissionApiImpl implements PermissionApi {
        @Override
        public QueryableRequestBuilder<DiPermission> get() {
            return new QueryableRequestBuilderImpl<>(new GenericType<>(DiPermission.class), () -> clients.get().path("/permission"));
        }
    }

    private final class SegmentationItemApiImpl extends ItemCrudApiImpl<DiSegmentation, DiSegmentation> implements SegmentationItemApi {

        private SegmentationItemApiImpl(final String path, final Class<DiSegmentation> type,
                                        final Supplier<WebClient> clients) {
            super(path, type, clients);
        }

        @Override
        public SegmentApi segments() {
            return new SegmentApiImpl(path);
        }
    }

    private final class SegmentationApiImpl extends ItemsCrudApiImpl<DiSegmentation, DiSegmentation, SegmentationItemApi>
            implements Dispenser.SegmentationApi {

        private SegmentationApiImpl() {
            super("/segmentations", DiSegmentation.class, clients);
        }

        @Override
        protected GenericType<DiListResponse<DiSegmentation>> getListResponseType() {
            return new GenericType<DiListResponse<DiSegmentation>>() {
            };
        }

        @NotNull
        @Override
        public SegmentationItemApi byKey(final String key) {
            return new SegmentationItemApiImpl(basePath + "/" + key, type, clients);
        }
    }

    private final class SegmentApiImpl extends ItemsCrudApiImpl<DiSegment, DiSegment, ItemCrudApi<DiSegment, DiSegment>>
            implements Dispenser.SegmentApi {

        private SegmentApiImpl(final String basePath) {
            super(basePath + "/segments", DiSegment.class, clients);
        }

        @Override
        protected GenericType<DiListResponse<DiSegment>> getListResponseType() {
            return new GenericType<DiListResponse<DiSegment>>() {
            };
        }

        @NotNull
        @Override
        public Dispenser.ItemCrudApi<DiSegment, DiSegment> byKey(final String key) {
            return new SegmentItemCrudApiImpl<>(basePath + "/" + key, type, clients);
        }
    }

    private static final class SegmentItemCrudApiImpl<T, B> extends ItemCrudApiImpl<T, B> {

        private SegmentItemCrudApiImpl(final String path, final Class<T> type, final Supplier<WebClient> clients) {
            super(path, type, clients);
        }

        @TestOnly
        @Override
        public @NotNull CreateRequestBuilder<T> update(final B object) {
            return new CreateRequestBuilder<T>() {
                private String reqId;

                @NotNull
                @Override
                public T performBy(@NotNull final DiPerson person) {
                    return WebClientUtils.authenticatePerson(clients.get(), person)
                            .path(path)
                            .query("reqId", WebClientUtils.actualReqId(reqId))
                            .header("X-Req-Id", WebClientUtils.actualReqId(reqId))
                            .invoke("PATCH", object, type);
                }

                @Override
                public @NotNull CreateRequestBuilder<T> withReqId(@NotNull final String reqId) {
                    this.reqId = reqId;
                    return this;
                }
            };
        }
    }

    private static final class ResourseSegmentationApiImpl extends ItemCrudApiImpl<List<DiSegmentation>, List<DiSegmentation>>
            implements Dispenser.ResourceSegmentaionApi {
        private ResourseSegmentationApiImpl(final Supplier<WebClient> clients, final String path) {
            super(path + "/segmentations", new GenericType<List<DiSegmentation>>() {
            }, clients);
        }
    }

    private final class ResourceGroupApiImpl
            extends ItemsCrudApiImpl<DiResourceGroup, DiResourceGroup, ItemCrudApi<DiResourceGroup, DiResourceGroup>>
            implements Dispenser.ResourceGroupApi {

        private ResourceGroupApiImpl(final Supplier<WebClient> clients, final String basePath) {
            super(basePath + "/resource-groups", DiResourceGroup.class, clients);
        }

        @Override
        protected GenericType<DiListResponse<DiResourceGroup>> getListResponseType() {
            return new GenericType<DiListResponse<DiResourceGroup>>() {
            };
        }

        @NotNull
        @Override
        public ItemCrudApi<DiResourceGroup, DiResourceGroup> byKey(final String key) {
            return new ItemCrudApiImpl<>(basePath + "/" + key, type, clients);
        }
    }

    private final class ChecksApiImpl extends ItemsCrudApiImpl<DiCheck, DiCheck.Body, ItemCrudApi<DiCheck, DiCheck.Body>> {

        protected ChecksApiImpl(final String basePath, final Supplier<WebClient> clients) {
            super(basePath + "/checks", DiCheck.class, clients);
        }

        @Override
        protected GenericType<DiListResponse<DiCheck>> getListResponseType() {
            return new GenericType<DiListResponse<DiCheck>>() {
            };
        }

        @NotNull
        @Override
        public ItemCrudApi<DiCheck, DiCheck.Body> byKey(final String key) {
            return new ItemCrudApiImpl<>(basePath + "/" + key, type, clients);
        }
    }

    @NotNull
    @Override
    public Dispenser.ResourceApi resource() {
        return new ResourceApiImpl();
    }

    private final class ResourceApiImpl implements ResourceApi {
        private final String path = "/resources";

        @Override
        public QueryableRequestBuilder<DiListResponse<DiResource>> get() {
            return new QueryableRequestBuilderImpl<>(new GenericType<DiListResponse<DiResource>>() {
            }, () -> clients.get().path(path));
        }
    }

    @NotNull
    @Override
    public Dispenser.BotSyncStatusApi botSyncStatus() {
        return new BotSyncApiImpl();
    }

    private final class BotSyncApiImpl implements BotSyncStatusApi {
        private final String path = "/bot-sync-status";

        @Override
        public QueryableRequestBuilder<DiBotSyncStatus> get() {
            return new QueryableRequestBuilderImpl<>(new GenericType<DiBotSyncStatus>() {
            }, () -> clients.get().path(path));
        }
    }
}