package ru.yandex.qe.dispenser.client.v1.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.ws.rs.core.Response;

import org.apache.cxf.jaxrs.client.WebClient;
import org.apache.cxf.message.Message;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DiRequest;
import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.client.RequestBuilder;
import ru.yandex.qe.dispenser.client.v1.DiOAuthToken;
import ru.yandex.qe.dispenser.client.v1.DispenserFactory;
import ru.yandex.qe.dispenser.client.v1.impl.BatchQuotaChangeRequestBuilderImpl.BatchChangeRequestBody;
import ru.yandex.qe.dispenser.client.v1.impl.BatchQuotaChangeRequestBuilderImpl.BatchChangeResponseBody;

public final class MockDispenserFactory<T> extends RemoteDispenserFactory {
    @NotNull
    private final List<PerformMock<?>> mocks;

    public MockDispenserFactory(@NotNull final Builder builder) {
        super(config());
        mocks = builder.mocks;
    }

    @NotNull
    private static DispenserConfig config() {
        return new DispenserConfig()
                .setClientId("DISPENSER")
                .setEnvironment(DispenserConfig.Environment.DEVELOPMENT)
                .setDispenserHost("")
                .setServiceZombieOAuthToken(DiOAuthToken.of(""));
    }

    @NotNull
    public static MockDispenserFactory.Builder builder() {
        return new MockDispenserFactory.Builder();
    }

    public void set(@NotNull final PerformMock<T> mock) {
        mocks.clear();
        mocks.add(mock);
    }

    public void set(@NotNull final Class<? extends RequestBuilder<T>> performerClass, @NotNull final PerformMock<T> mock) {
        set(mock);
    }

    @NotNull
    @Override
    WebClient createUnconfiguredClient(@NotNull final String address) {
        return new PatchedWebClient(address) {
            @Override
            protected void doRunInterceptorChain(@NotNull final Message m) {
                try {
                    doIntercept(m);
                } catch (Exception e) {
                    m.setContent(Exception.class, e);
                }
            }

            private void doIntercept(@NotNull final Message m) throws Exception {
                final DiRequest<?> req = new DiRequestImpl<>(m);
                for (final PerformMock<?> mock : mocks) {
                    final Object responseBody = mock.getResponse(req);
                    if (responseBody == null) {
                        continue;
                    }
                    final Response response = responseBody instanceof Response ? (Response) responseBody
                            : Response.ok(responseBody).build();
                    m.getExchange().put(List.class, Collections.singletonList(response));
                    m.getExchange().put(Message.RESPONSE_CODE, response.getStatus());
                    m.getExchange().put("IN_CHAIN_COMPLETE", Boolean.TRUE);
                    return;
                }
                throw new AssertionError("No case configured in MockDispenserFactory for this request: " + req);
            }
        };
    }

    public static final class Builder implements DtoBuilder<DispenserFactory> {
        @NotNull
        private final List<PerformMock<?>> mocks = new ArrayList<>();

        private Builder() {
        }

        @NotNull
        public Builder mock(@NotNull final PerformMock<?> mock, @NotNull final Runnable callback) {
            mocks.add(req -> {
                Object resp = null;
                try {
                    return resp = mock.getResponse(req);
                } finally {
                    if (resp != null) {
                        callback.run();
                    }
                }
            });
            return this;
        }

        @NotNull
        public Builder mock(@NotNull final Object requestBody, @NotNull final Object responseBody, @NotNull final Runnable callback) {
            return mock((req) -> requestBody.equals(req.getBody()) ? responseBody : null, callback);
        }

        @NotNull
        public Builder mock(@NotNull final String pathWithQuery, @NotNull final Object responseBody, @NotNull final Runnable callback) {
            return mock((req) -> pathWithQuery.equals(req.getFullUrl()) ? responseBody : null, callback);
        }

        @NotNull
        public Builder timeout(final long sleepBeforeTimeoutInMillis, @NotNull final Runnable callback) {
            return mock(new DiMocks.Timeout<>(sleepBeforeTimeoutInMillis), callback);
        }

        @NotNull
        public Builder timeout(@NotNull final String reqId, final long sleepBeforeTimeoutInMillis, @NotNull final Runnable callback) {
            return mock(ifReqId(reqId, new DiMocks.Timeout<>(sleepBeforeTimeoutInMillis)), callback);
        }

        @NotNull
        public Builder timeout(@NotNull final DiMocks.ChangeQuotaRequest request, final long sleepBeforeTimeoutInMillis, @NotNull final Runnable callback) {
            return mock(ifRequest(request, new DiMocks.Timeout<>(sleepBeforeTimeoutInMillis)), callback);
        }

        @NotNull
        public Builder error(@NotNull final Runnable callback) {
            return mock(new DiMocks.ServerError(), callback);
        }

        @NotNull
        public Builder error(@NotNull final String reqId, @NotNull final Runnable callback) {
            return mock(ifReqId(reqId, new DiMocks.ServerError()), callback);
        }

        @NotNull
        public Builder error(@NotNull final DiMocks.ChangeQuotaRequest request, @NotNull final Runnable callback) {
            return mock(ifRequest(request, new DiMocks.ServerError()), callback);
        }

        @NotNull
        public Builder success(@NotNull final Runnable callback) {
            return mock(this::successResponse, callback);
        }

        @NotNull
        public Builder success(@NotNull final String reqId, @NotNull final Runnable callback) {
            return mock(ifReqId(reqId, this::successResponse), callback);
        }

        @NotNull
        public Builder success(@NotNull final DiMocks.ChangeQuotaRequest request, @NotNull final Runnable callback) {
            return mock(ifRequest(request, this::successResponse), callback);
        }

        @NotNull
        private <T> PerformMock<T> ifReqId(@NotNull final String reqId, @NotNull final PerformMock<T> then) {
            return req -> reqId.equals(req.getQueryParam("reqId")) ? then.getResponse(req) : null;
        }

        @NotNull
        private <T> PerformMock<T> ifRequest(@NotNull final DiMocks.ChangeQuotaRequest request, @NotNull final PerformMock<T> then) {
            return req -> request.delegate.equals(req.getBody()) ? then.getResponse(req) : null;
        }

        @NotNull
        private BatchChangeResponseBody successResponse(@NotNull final DiRequest<?> request) {
            final int operationsCount = Objects.requireNonNull((BatchChangeRequestBody) request.getBody()).size();
            final List<BatchChangeResponseBody.OperationResultReference> operationResults = IntStream.rangeClosed(1, operationsCount)
                    .mapToObj(i -> new BatchChangeResponseBody.OperationResultReference(i, true, null))
                    .collect(Collectors.toList());
            return new BatchChangeResponseBody(operationResults);
        }

        @NotNull
        @Override
        public DispenserFactory build() {
            return new MockDispenserFactory<>(this);
        }
    }

    public interface PerformMock<T> {
        @Nullable
        T getResponse(@NotNull final DiRequest<?> req) throws Exception;
    }
}
