package ru.yandex.qe.dispenser.client.v1.impl;

import java.util.List;
import java.util.function.Supplier;

import javax.ws.rs.HttpMethod;
import javax.ws.rs.core.GenericType;

import org.apache.cxf.jaxrs.client.WebClient;
import org.jetbrains.annotations.NotNull;

import org.jetbrains.annotations.Nullable;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequest;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeStatistic;
import ru.yandex.qe.dispenser.api.v1.request.quota.Body;
import ru.yandex.qe.dispenser.api.v1.request.quota.BodyUpdate;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.client.QueryableRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.DiPerson;
import ru.yandex.qe.dispenser.client.v1.Dispenser;
import ru.yandex.qe.dispenser.client.v1.builder.CreateRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.builder.SimplePersonRequestBuilder;

class QuotaChangeRequestApiImpl
        implements Dispenser.QuotaChangeRequestApi {

    private static final Class<DiQuotaChangeRequest> TYPE = DiQuotaChangeRequest.class;
    private static final GenericType<DiListResponse<DiQuotaChangeRequest>> LIST_TYPE = new GenericType<DiListResponse<DiQuotaChangeRequest>>() {
    };
    public static final GenericType<DiQuotaChangeStatistic> STATISTIC_TYPE = new GenericType<>(DiQuotaChangeStatistic.class);
    private static final String BASE_PATH = "/quota-requests";

    private final Supplier<WebClient> clients;
    private final String basePath;

    protected QuotaChangeRequestApiImpl(final Supplier<WebClient> clients) {
        this.clients = clients;
        this.basePath = BASE_PATH;
    }

    @NotNull
    @Override
    public CreateRequestBuilder<DiListResponse<DiQuotaChangeRequest>> create(final Body itemBody, @Nullable Long campaignId) {
        return new CreateRequestBuilder<DiListResponse<DiQuotaChangeRequest>>() {
            private String reqId;

            @NotNull
            @Override
            public DiListResponse<DiQuotaChangeRequest> performBy(@NotNull final DiPerson person) {
                WebClient requestBuilder = WebClientUtils.authenticatePersonWithReqId(clients.get(), person, reqId)
                        .path(basePath);
                if (campaignId != null) {
                    requestBuilder.replaceQueryParam("campaign", campaignId);
                }
                return requestBuilder.post(itemBody, LIST_TYPE);
            }

            @Override
            public @NotNull CreateRequestBuilder<DiListResponse<DiQuotaChangeRequest>> withReqId(@NotNull final String reqId) {
                this.reqId = reqId;
                return this;
            }
        };
    }

    @Override
    public @NotNull CreateRequestBuilder<DiListResponse<DiQuotaChangeRequest>> batchCreate(
            final List<Body> itemBodies, @Nullable Long campaignId) {
        return new CreateRequestBuilder<DiListResponse<DiQuotaChangeRequest>>() {
            private String reqId;

            @NotNull
            @Override
            public DiListResponse<DiQuotaChangeRequest> performBy(@NotNull final DiPerson person) {
                WebClient requestBuilder = WebClientUtils.authenticatePersonWithReqId(clients.get(), person, reqId)
                        .path(basePath + "/batch");
                if (campaignId != null) {
                    requestBuilder.replaceQueryParam("campaign", campaignId);
                }
                return requestBuilder.post(itemBodies, LIST_TYPE);
            }

            @Override
            public @NotNull CreateRequestBuilder<DiListResponse<DiQuotaChangeRequest>> withReqId(@NotNull final String reqId) {
                this.reqId = reqId;
                return this;
            }
        };
    }

    @NotNull
    @Override
    public Dispenser.QuotaChangeRequestItemApi byId(final long id) {
        return new QuotaItemChangeRequestApiImpl(basePath + "/" + id, TYPE, clients);
    }

    @NotNull
    @Override
    public QueryableRequestBuilder<DiListResponse<DiQuotaChangeRequest>> get() {
        return new QueryableRequestBuilderImpl<>(LIST_TYPE, () -> clients.get().path(basePath));
    }

    @NotNull
    @Override
    public QueryableRequestBuilder<DiQuotaChangeStatistic> getStatistic() {
        return new QueryableRequestBuilderImpl<>(STATISTIC_TYPE, () -> clients.get().path(basePath + "/statistics"));
    }

    public static class QuotaItemChangeRequestApiImpl extends ItemCrudApiImpl<DiQuotaChangeRequest, Body>
            implements Dispenser.QuotaChangeRequestItemApi {
        protected QuotaItemChangeRequestApiImpl(final String path, final Class<DiQuotaChangeRequest> type,
                                                final Supplier<WebClient> clients) {
            super(path, type, clients);
        }

        @Override
        @NotNull
        public SimplePersonRequestBuilder<DiQuotaChangeRequest> setStatus(final DiQuotaChangeRequest.Status status) {
            return (@NotNull final DiPerson person) -> {
                return WebClientUtils.authenticatePersonWithGeneratedReqId(clients.get(), person)
                        .path(path + "/status/" + status.name())
                        .put(null, type);
            };
        }

        @Override
        @NotNull
        public SimplePersonRequestBuilder<DiQuotaChangeRequest> setStatus(final DiQuotaChangeRequest.Status status,
                                                                          boolean suppressSummon) {
            return (@NotNull final DiPerson person) -> {
                return WebClientUtils.authenticatePersonWithGeneratedReqId(clients.get(), person)
                        .path(path + "/status/" + status.name())
                        .query("suppressSummon", suppressSummon)
                        .put(null, type);
            };
        }

        @Override
        @NotNull
        public SimplePersonRequestBuilder<DiQuotaChangeRequest> createTicket() {
            return (@NotNull final DiPerson person) -> {
                return WebClientUtils.authenticatePersonWithGeneratedReqId(clients.get(), person)
                        .path(path + "/create-ticket")
                        .post(null, type);
            };
        }

        @Override
        @NotNull
        public SimplePersonRequestBuilder<DiQuotaChangeRequest> update(final BodyUpdate bodyUpdate) {
            return (@NotNull final DiPerson person) -> {
                return WebClientUtils.authenticatePersonWithGeneratedReqId(clients.get(), person)
                        .path(path)
                        .invoke(HttpMethod.PATCH, bodyUpdate, type);
            };
        }

        @Override
        @NotNull
        public SimplePersonRequestBuilder<DiQuotaChangeRequest> update(final BodyUpdate bodyUpdate,
                                                                       boolean suppressSummon) {
            return (@NotNull final DiPerson person) -> {
                return WebClientUtils.authenticatePersonWithGeneratedReqId(clients.get(), person)
                        .path(path)
                        .query("suppressSummon", suppressSummon)
                        .invoke(HttpMethod.PATCH, bodyUpdate, type);
            };
        }

    }
}
