package ru.yandex.qe.bus.exception;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.ws.rs.InternalServerErrorException;
import javax.ws.rs.WebApplicationException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;
import static java.util.stream.Stream.of;

/**
 * @author rurikk
 */
@SuppressWarnings("unchecked")
public abstract class AbstractExceptionTranslator implements Function<Throwable, WebApplicationException> {
    /**
     * Non-static logger to report on behalf of subclass.
     */
    protected Logger log = LoggerFactory.getLogger(getClass());

    private Map<Class, Function<Throwable, WebApplicationException>> mapper = new HashMap<>();

    protected AbstractExceptionTranslator() {
        on(Throwable.class)
                .logErrorMessage()
                .wrap(InternalServerErrorException::new);
    }

    protected <E extends Throwable> Action<E> on(Class<? extends E> exceptionClass, Class<? extends E>... moreClasses) {
        return new Action<>(concat(of(exceptionClass), of(moreClasses)).collect(toList()));
    }

    protected <E extends WebApplicationException> WaeAction<E> onWae(Class<? extends E> exceptionClass, Class<? extends E>... moreClasses) {
        return new WaeAction<>(concat(of(exceptionClass), of(moreClasses)).collect(toList()));
    }

    @Override
    public WebApplicationException apply(Throwable throwable) {
        return findMostSpecific(throwable.getClass())
                .apply(throwable);
    }

    private Function<Throwable, WebApplicationException> findMostSpecific(Class<? extends Throwable> clazz) {
        Function<Throwable, WebApplicationException> res = mapper.get(clazz);
        return res != null ? res : findMostSpecific((Class) clazz.getSuperclass());
    }

    protected class Action<E extends Throwable> {
        private final List<Class<? extends E>> exceptionClasses;
        private Consumer<E> consumer = e -> {
        };

        private Action(List<Class<? extends E>> classes) {
            this.exceptionClasses = classes;
        }

        public Action<E> peek(Consumer<E> consumer) {
            this.consumer = this.consumer.andThen(consumer);
            return this;
        }

        public Action<E> logInfoMessage() {
            return peek(e -> log.info(e.toString()));
        }

        public Action<E> logInfo(String message) {
            return peek(e -> log.info(message));
        }

        public Action<E> logWarnMessage() {
            return peek(e -> log.warn(e.getMessage(), e));
        }

        public Action<E> logWarn(String message) {
            return peek(e -> log.warn(message, e));
        }

        public Action<E> logErrorMessage() {
            return peek(e -> log.error(e.getMessage(), e));
        }

        public Action<E> logError(String message) {
            return peek(e -> log.error(message, e));
        }


        public void wrap(Function<E, WebApplicationException> translate) {
            exceptionClasses.forEach(ec ->
                    mapper.put(ec, (Function) (Function<E, WebApplicationException>) (E e) -> {
                        consumer.accept(e);
                        return translate.apply(e);
                    }));
        }
    }

    protected class WaeAction<E extends WebApplicationException> extends Action<E> {
        private WaeAction(List<Class<? extends E>> classes) {
            super(classes);
        }

        @Override
        public WaeAction<E> peek(Consumer<E> consumer) {
            return (WaeAction<E>) super.peek(consumer);
        }

        @Override
        public WaeAction<E> logInfoMessage() {
            return (WaeAction<E>) super.logInfoMessage();
        }

        @Override
        public WaeAction<E> logInfo(String message) {
            return (WaeAction<E>) super.logInfo(message);
        }

        @Override
        public WaeAction<E> logWarnMessage() {
            return (WaeAction<E>) super.logWarnMessage();
        }

        @Override
        public WaeAction<E> logWarn(String message) {
            return (WaeAction<E>) super.logWarn(message);
        }

        @Override
        public WaeAction<E> logErrorMessage() {
            return (WaeAction<E>) super.logErrorMessage();
        }

        @Override
        public WaeAction<E> logError(String message) {
            return (WaeAction<E>) super.logError(message);
        }

        public void rethrow() {
            wrap(e -> e);
        }
    }

}
