package ru.yandex.qe.bus.exception;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.BadRequestException;
import javax.ws.rs.InternalServerErrorException;
import javax.ws.rs.NotAuthorizedException;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.core.Response;

import static ru.yandex.qe.bus.ResponseUtils.newResponse;

public abstract class ExceptionUtils {

    @SuppressWarnings("UnusedDeclaration")
    public static void checkNotNullParameter(@Nullable Object object,
                                             @Nonnull String message) {
        if (object == null) {
            throw newBadRequestException(message, object);
        }
    }

    @SuppressWarnings("UnusedDeclaration")
    public static void checkStateParameter(boolean state,
                                           @Nonnull String message,
                                           @Nullable Object object) {
        if (!(state)) {
            throw newBadRequestException(message, object);
        }
    }

    @SuppressWarnings("UnusedDeclaration")
    public static void checkNotEmptyParameter(@Nullable String object,
                                              @Nonnull String message) {
        if ((object == null) || (object.trim().isEmpty())) {
            throw newBadRequestException(message, object);
        }
    }

    @SuppressWarnings("UnusedDeclaration")
    public static <T> T throwNotFoundIfNull(T object, @Nonnull String message, @Nonnull Object... args) {
        if (object == null) {
            throw newNotFoundException(message, args);
        }
        return object;
    }

    @SuppressWarnings("UnusedDeclaration")
    @Nonnull
    public static BadRequestException newBadRequestException(@Nonnull String message,
                                                             @Nonnull Object... args) {
        String text = formatMessage(message, args);
        return new BadRequestException(
                text,
                newResponse(
                        Response.Status.BAD_REQUEST,
                        text));
    }

    @SuppressWarnings("UnusedDeclaration")
    @Nonnull
    public static BadRequestException newBadRequestException(@Nonnull Throwable cause,
                                                             @Nonnull String message,
                                                             @Nonnull Object... args) {
        String text = formatMessage(message, args);
        return new BadRequestException(
                text,
                newResponse(
                        Response.Status.BAD_REQUEST,
                        text),
                cause);
    }

    @SuppressWarnings("UnusedDeclaration")
    @Nonnull
    public static NotFoundException newNotFoundException(@Nonnull String message,
                                                         @Nonnull Object... args) {
        String text = formatMessage(message, args);
        return new NotFoundException(
                text,
                newResponse(
                        Response.Status.NOT_FOUND,
                        text));
    }

    @SuppressWarnings("UnusedDeclaration")
    @Nonnull
    public static NotFoundException newNotFoundException(@Nonnull Throwable cause,
                                                         @Nonnull String message,
                                                         @Nonnull Object... args) {
        String text = formatMessage(message, args);
        return new NotFoundException(
                text,
                newResponse(
                        Response.Status.NOT_FOUND,
                        text),
                cause);
    }

    @SuppressWarnings("UnusedDeclaration")
    @Nonnull
    public static NotAuthorizedException newNotAuthorizedException(@Nonnull String message,
                                                                   @Nonnull Object... args) {
        String text = formatMessage(message, args);
        return new NotAuthorizedException(
                text,
                newResponse(
                        Response.Status.UNAUTHORIZED,
                        text));
    }

    @SuppressWarnings("UnusedDeclaration")
    @Nonnull
    public static NotAuthorizedException newNotAuthorizedException(@Nonnull Throwable cause,
                                                                   @Nonnull String message,
                                                                   @Nonnull Object... args) {
        String text = formatMessage(message, args);
        return new NotAuthorizedException(
                text,
                cause,
                newResponse(
                        Response.Status.UNAUTHORIZED,
                        text));
    }

    @SuppressWarnings("UnusedDeclaration")
    @Nonnull
    public static InternalServerErrorException newInternalServerErrorException(@Nonnull String message,
                                                                               @Nonnull Object... args) {
        String text = formatMessage(message, args);
        return new InternalServerErrorException(
                text,
                newResponse(
                        Response.Status.INTERNAL_SERVER_ERROR,
                        text));
    }

    @SuppressWarnings("UnusedDeclaration")
    @Nonnull
    public static InternalServerErrorException newInternalServerErrorException(@Nonnull Throwable cause,
                                                                               @Nonnull String message,
                                                                               @Nonnull Object... args) {
        String text = formatMessage(message, args);
        return new InternalServerErrorException(
                text,
                newResponse(
                        Response.Status.INTERNAL_SERVER_ERROR,
                        text),
                cause);
    }

    @Nonnull
    public static String getExceptionMessage(@Nullable Throwable throwable) {
        if (throwable == null) {
            return "null";
        }
        StringBuilder result = new StringBuilder();
        for (Throwable i = throwable; i != null; i = i.getCause()) {
            result.append(i);
            if (i.getStackTrace().length > 0) {
                result.append(" - ");
                result.append(i.getStackTrace()[0]);
            }
            if (i.getCause() != null) {
                result.append(" <- ");
            }
        }
        return result.toString();
    }

    @Nonnull
    private static String formatMessage(@Nonnull String message, @Nonnull Object... args) {
        return args.length > 0 ? String.format(message, args) : message;
    }
}
