package ru.yandex.qe.bus.factories;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.ws.rs.ext.ExceptionMapper;
import javax.ws.rs.ext.MessageBodyReader;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.ParamConverterProvider;

import org.apache.cxf.Bus;
import org.apache.cxf.jaxrs.AbstractJAXRSFactoryBean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

public abstract class BusEndpointBuilder implements ApplicationContextAware {

    private static final Logger LOG = LoggerFactory.getLogger(BusEndpointBuilder.class);

    private String transportId;

    protected String address;

    private List<?> providers;

    private Bus bus;

    private boolean enableExceptionMapperAutodiscovery = true;

    protected ApplicationContext ctx;

    public void setTransportId(@Nonnull String transportId) {
        this.transportId = transportId;
    }

    public void setAddress(@Nonnull String address) {
        this.address = address;
    }

    public void setProviders(@Nonnull List<?> providers) {
        this.providers = providers;
    }

    public void setBus(@Nonnull Bus bus) {
        this.bus = bus;
    }

    public void setEnableExceptionMapperAutodiscovery(final boolean enableExceptionMapperAutodiscovery) {
        this.enableExceptionMapperAutodiscovery = enableExceptionMapperAutodiscovery;
    }

    protected void prepareFactoryBean(AbstractJAXRSFactoryBean jaxrsFactoryBean) {
        jaxrsFactoryBean.setTransportId(transportId);
        jaxrsFactoryBean.setAddress(address);
        jaxrsFactoryBean.setProviders(discoverProviders());
        jaxrsFactoryBean.setBus(bus);
        //cxf, don't set bus features to endpoint factories,
        //to prevent desync features in xml definitions for bus and factories I prefer force traverse
        jaxrsFactoryBean.setFeatures(new ArrayList<>(bus.getFeatures()));
    }

    protected void prepareFactoryBean(AbstractJAXRSFactoryBean jaxrsFactoryBean, String address) {
        jaxrsFactoryBean.setTransportId(transportId);
        jaxrsFactoryBean.setAddress(address);
        jaxrsFactoryBean.setProviders(providers);
        jaxrsFactoryBean.setBus(bus);
        //cxf, don't set bus features to endpoint factories,
        //to prevent desync features in xml definitions for bus and factories I prefer force traverse
        jaxrsFactoryBean.setFeatures(new ArrayList<>(bus.getFeatures()));
    }

    public Bus getBus() {
        return bus;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.ctx = applicationContext;
    }

    private List<?> discoverProviders() {
        Set allProviders = new HashSet<>(providers);

        allProviders.addAll(ctx.getBeansOfType(MessageBodyWriter.class).values());
        allProviders.addAll(ctx.getBeansOfType(MessageBodyReader.class).values());
        allProviders.addAll(ctx.getBeansOfType(ParamConverterProvider.class).values());
        if (enableExceptionMapperAutodiscovery) {
            allProviders.addAll(ctx.getBeansOfType(ExceptionMapper.class).values());
        }

        LOG.debug("Using the following REST providers: {}", allProviders);

        return new ArrayList(allProviders);
    }
}
