package ru.yandex.qe.bus.javatime;

import java.time.DateTimeException;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.TemporalAccessor;

import javax.ws.rs.ext.ParamConverter;

/**
 * @author Alexei Zakharov (ayza)
 */
public class ZonedDateTimeParamConverter implements ParamConverter<ZonedDateTime> {
  static final ZoneId DEFAULT_ZONE = ZoneId.systemDefault();

  static final DateTimeFormatter FORMATTER = new DateTimeFormatterBuilder()
      .append(DateTimeFormatter.ISO_LOCAL_DATE)
      .optionalStart()
      .appendLiteral('T')
      .append(DateTimeFormatter.ISO_TIME)
      .toFormatter();

  @Override
  public ZonedDateTime fromString(String value) {
    try {
      TemporalAccessor obj = FORMATTER.parseBest(value, ZonedDateTime::from, LocalDate::from);
      if (obj instanceof LocalDate) {
        return withZeroTimeAndDefaultTz((LocalDate) obj);
      } else if (obj instanceof ZonedDateTime) {
        return (ZonedDateTime) obj;
      } else {
        throw new IllegalArgumentException("Unable to parse date-time: " + value);
      }
    } catch (DateTimeException e) {
      // let try basic ISO date or throw exception if doesn't match
      return withZeroTimeAndDefaultTz(DateTimeFormatter.BASIC_ISO_DATE.parse(value, LocalDate::from));
    }
  }

  private ZonedDateTime withZeroTimeAndDefaultTz(LocalDate date) {
    return ZonedDateTime.of(date, LocalTime.ofSecondOfDay(0), DEFAULT_ZONE);
  }


  @Override
  public String toString(ZonedDateTime value) {
    return FORMATTER.format(value);
  }
}
