package ru.yandex.qe.http.certificates;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Class used to provide set of x509 trust managers with support for custom yandex CA and default system certificates.
 *
 * @author nkey
 * @since 17.03.14
 */
public final class TrustManagersProvider {

    private static final Logger LOGGER = LoggerFactory.getLogger(TrustManagersProvider.class);
    private static final String JSK_YANDEX_CA_STORE = "certificates/YandexInternalCA.jks";
    private static final String STORE_PASSWORD = "yandex";

    private TrustManager[] trustManagers;

    private TrustManagersProvider() {
        try {
            List<TrustManager> customTrustManagers = getCustomTrustManagers();
            List<TrustManager> defaultTrustManagers = getTrustManagersForKeyStore(null);

            final Optional<X509TrustManager> customX509 = getX509TrustManager(customTrustManagers);
            final Optional<X509TrustManager> defaultX509 = getX509TrustManager(defaultTrustManagers);

            X509TrustManager resultTrustManager = createCompositeTrustManager(customX509, defaultX509);

            Iterable<TrustManager> all = Iterables
                    .concat(customTrustManagers, defaultTrustManagers, Collections.singleton(resultTrustManager));

            trustManagers = FluentIterable.from(all).filter(new Predicate<TrustManager>() {
                @Override
                public boolean apply(TrustManager input) {
                    return input != customX509.get() && input != defaultX509.get();
                }
            }).toArray(TrustManager.class);


        } catch (NoSuchAlgorithmException | KeyStoreException | CertificateException | IOException e) {
            String msg = "Can't init yandex root CA setting";
            LOGGER.error(msg, e);
            throw new RuntimeException(msg, e);
        }
    }

    private X509TrustManager createCompositeTrustManager(Optional<X509TrustManager> customX509,
                                                         Optional<X509TrustManager> defaultX509) {
        X509TrustManager resultTrustManager;
        if (!customX509.isPresent() && !defaultX509.isPresent()) {
            String msg = "Can't get any x590 trust manager";
            LOGGER.error(msg);
            throw new IllegalStateException(msg);
        } else if (customX509.isPresent() && defaultX509.isPresent()) {
            resultTrustManager = new CompositeX509TrustManager(customX509.get(), defaultX509.get());
        } else {
            resultTrustManager = customX509.or(defaultX509.orNull());
        }
        return resultTrustManager;
    }

    private List<TrustManager> getCustomTrustManagers()
            throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException {
        KeyStore keyStore = KeyStore.getInstance("jks");
        keyStore.load(TrustManagersProvider.class.getClassLoader().getResourceAsStream(JSK_YANDEX_CA_STORE),
                STORE_PASSWORD.toCharArray());
        return getTrustManagersForKeyStore(keyStore);
    }

    private List<TrustManager> getTrustManagersForKeyStore(KeyStore keyStore)
            throws NoSuchAlgorithmException, KeyStoreException {
        TrustManagerFactory trustManagerFactory =
                TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        trustManagerFactory.init(keyStore);

        return Arrays.asList(trustManagerFactory.getTrustManagers());
    }

    private Optional<X509TrustManager> getX509TrustManager(List<TrustManager> customTrustManagers) {
        return FluentIterable.from(customTrustManagers).filter(X509TrustManager.class).first();
    }

    private static final class LazyHolder {
        private static final TrustManagersProvider instance = new TrustManagersProvider();
    }

    public static TrustManagersProvider getInstance() {
        return LazyHolder.instance;
    }

    public TrustManager[] getTrustManagers() {
        return trustManagers;
    }

    public SSLContext createSSLContext() {
        SSLContext context;
        try {
            context = SSLContext.getInstance("TLS");
            context.init(null, trustManagers, null);
            return context;
        } catch (NoSuchAlgorithmException | KeyManagementException e) {
            String msg = "Error on creating SSL context";
            LOGGER.error(msg, e);
            throw new IllegalStateException(msg, e);
        }
    }

    public TrustManager[] trustAllManagers() {
        X509TrustManager trustManager = new X509TrustManager() {
            @Override
            public void checkClientTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
            }

            @Override
            public void checkServerTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {
            }

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return new X509Certificate[0];
            }
        };
        return new TrustManager[]{ trustManager };
    }
}
