package ru.yandex.qe.ssl;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.security.cert.CRL;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.concurrent.Immutable;

import com.google.common.base.Preconditions;
import com.google.common.io.ByteStreams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Throwables.propagate;
import static java.lang.Math.min;

/**
 * Загрузчик Certificate Revocation List (CRL).
 * Загружает CRL с периодичностью 90-100% от {@link Configuration#normalRetry}.
 * <p/>
 * Первая загрузка происходит из файла {@link Configuration#file}. При невозможности прочитать корректный CRL из файла,
 * будет произведена попытка удаленной загрузки из {@link Configuration#url}, но перед ней будет произведена задержка
 * 0-100% от {@link Configuration#initialDelay} c целью снижения нагрузки на удаленный ресурс при одновременном старте
 * большого количества клиентов.
 * <p/>
 * При неудачной попытке удаленной загрузки повторная попытка будет осуществлена через {@link Configuration#minRetry},
 * затем промежуток будет увеличиваться до достижения {@link Configuration#maxRetry}.
 * Для распределения нагрузки к полученым интервалам применяется коэффициент 50-100%.
 * После получения корректного файла интервал будет возобновлен до {@link Configuration#normalRetry}.
 * <p/>
 * Каждая успешная загрузка из удаленного источника сопровождается записью в тот же файл {@link Configuration#file},
 * из которого загрузчик читает на старте.
 *
 * {@link CrlFetcher#sync} позволяет дождаться, пока CRL будет загружен либо из файла, либо удаленно. Время ожидания
 * регулируется {@link Configuration#syncTimeout}
 *
 * @author rurikk
 */
public class CrlFetcher {
    private static final Logger log = LoggerFactory.getLogger(CrlFetcher.class);

    private final ScheduledExecutorService executor;
    private final Clock clock;
    private Configuration configuration;
    private volatile X509CRL crl;
    private CompletableFuture<X509CRL> loaded = new CompletableFuture<>();
    private Optional<Collection<X509Certificate>> trustedCertificates = Optional.empty();


    public CrlFetcher(ScheduledExecutorService executor, Configuration configuration, Clock clock) {
        this.executor = executor;
        this.configuration = configuration;
        this.clock = clock;
    }

    public Configuration getConfiguration() {
        return configuration;
    }

    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    public void setTrustedCertificates(Collection<X509Certificate> trustedCertificates) {
        this.trustedCertificates = Optional.of(trustedCertificates);
    }

    public CrlFetcher(Configuration configuration) {
        this(Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, CrlFetcher.class.getSimpleName())), configuration, Clock.systemDefaultZone());
    }

    public CrlFetcher start() {
        executor.submit(new Work());
        return this;
    }

    public void stop() {
        executor.shutdown();
    }

    /**
     * Waits until CRL first time loaded.
     */
    public CrlFetcher sync() {
        try {
            loaded.get(configuration.syncTimeout.toMillis(), TimeUnit.MILLISECONDS);
            Preconditions.checkNotNull(crl);
        } catch (Exception e) {
            throw propagate(e);
        }
        return this;
    }

    public Iterable<CRL> getCrls() {
        return Optional.<CRL>ofNullable(crl)
                .map(Collections::singletonList)
                .orElse(Collections.emptyList());
    }

    @Immutable
    public static class Configuration {
        private final File file;
        private final URL url;
        private final Duration initialDelay;
        private final Duration normalRetry;
        private final Duration minRetry;
        private final Duration maxRetry;
        private final Duration syncTimeout;

        public Configuration(File file, URL url, Duration initialDelay, Duration normalRetry, Duration minRetry, Duration maxRetry) {
            this(file, url, initialDelay, normalRetry, minRetry, maxRetry, Duration.ofSeconds(10));
        }

        public Configuration(File file, URL url, Duration initialDelay, Duration normalRetry, Duration minRetry, Duration maxRetry, Duration syncTimeout) {
            this.url = url;
            this.file = file;
            this.initialDelay = initialDelay;
            this.normalRetry = normalRetry;
            this.minRetry = minRetry;
            this.maxRetry = maxRetry;
            this.syncTimeout = syncTimeout;
        }

        public File getFile() {
            return file;
        }

        public URL getUrl() {
            return url;
        }

        public Duration getInitialDelay() {
            return initialDelay;
        }

        public Duration getNormalRetry() {
            return normalRetry;
        }

        public Duration getMinRetry() {
            return minRetry;
        }

        public Duration getMaxRetry() {
            return maxRetry;
        }

        public Duration getSyncTimeout() {
            return syncTimeout;
        }
    }

    class Work implements Runnable {
        Random r = new Random();
        boolean firstRun = true;
        int attempt = 0;

        @Override
        public void run() {
            X509CRL newCrl = firstRun ? tryLoadFromFile(configuration.file) : tryLoadFromUrl(configuration.url);
            if (newCrl != null) {
                crl = newCrl;
            }
            notifyWaiting();
            boolean crlIsOk = newCrl != null
                    && newCrl.getNextUpdate().toInstant().isAfter(Instant.now(clock).plus(configuration.normalRetry));
            long delay = delayBeforeNextRetry(crlIsOk);
            firstRun = false;

            log.info("Scheduling CRL loading in {}", Duration.ofMillis(delay).toString());
            executor.schedule(this, delay, TimeUnit.MILLISECONDS);
        }

        private void verifyWithAnyTrusted(X509CRL crl) {
            trustedCertificates.ifPresent(certificates -> certificates.stream()
                    .filter(cert -> {
                        try {
                            crl.verify(cert.getPublicKey());
                            return true;
                        } catch (Exception e) {
                            return false;
                        }
                    })
                    .findAny()
                    .orElseThrow(() -> new IllegalArgumentException("Could not verify CRL with any trusted certificate")));
        }

        private void notifyWaiting() {
            if (crl != null) {
                loaded.complete(crl);
            }
        }

        private X509CRL tryLoadFromFile(File file) {
            log.info("Loading CRL from file: " + file);
            if (file.length() == 0) {
                log.info("CRL file is empty or doesn't exist: " + file);
                return null;
            }
            try (InputStream is = new FileInputStream(file)) {
                X509CRL crl = (X509CRL) CertificateFactory.getInstance("X.509").generateCRL(is);
                verifyWithAnyTrusted(crl);
                checkNextUpdate(crl);
                log.info("CRL successfully loaded from file");
                return crl;
            } catch (Exception e) {
                log.warn("CRL loading failed", e);
                return null;
            }
        }

        private X509CRL tryLoadFromUrl(URL url) {
            log.info("Loading CRL from url:" + url);
            try (InputStream is = url.openStream()) {
                byte[] bytes = ByteStreams.toByteArray(is);
                X509CRL crl = (X509CRL) CertificateFactory.getInstance("X.509").generateCRL(new ByteArrayInputStream(bytes));
                verifyWithAnyTrusted(crl);
                checkNextUpdate(crl);
                log.info("CRL successfully loaded from remote");
                tryCacheToFile(bytes);
                return crl;
            } catch (Exception e) {
                log.warn("CRL loading failed", e);
                return null;
            }
        }

        private void tryCacheToFile(byte[] bytes) {
            try {
                Files.write(configuration.file.toPath(), bytes);
            } catch (Exception e) {
                log.warn("Cannot cache CRL to file: " + configuration.file, e);
            }
        }

        private void checkNextUpdate(X509CRL crl) {
            Date nextUpdate = crl.getNextUpdate();
            if (nextUpdate.before(Date.from(Instant.now(clock)))) {
                throw new RuntimeException("CRL is expired, valid till " + nextUpdate);
            }
        }

        long delayBeforeNextRetry(boolean crlIsOk) {
            if (crlIsOk) {
                attempt = 0;
                return (long) (configuration.normalRetry.toMillis() * rnd(0.9, 1));
            } else if (firstRun) {
                attempt = 0;
                return (long) (configuration.initialDelay.toMillis() * rnd(0, 1));
            } else {
                ++attempt;
                return (long) min(
                        configuration.maxRetry.toMillis(),
                        configuration.minRetry.toMillis() * rnd(0.5, 1) * (1L << min(attempt, 30)));
            }
        }

        private double rnd(double min, double max) {
            return min + (max - min) * r.nextFloat();
        }
    }
}
