package ru.yandex.qe.http;

import java.io.File;
import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import com.google.common.base.Throwables;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.client.RedirectStrategy;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.protocol.HttpRequestExecutor;

import ru.yandex.qe.http.aspects.AspectCapableHttpClient;
import ru.yandex.qe.http.aspects.HttpAspectFactory;
import ru.yandex.qe.http.aspects.log.LogHttpAspectFactory;
import ru.yandex.qe.http.aspects.log.LogHttpRequestInterceptor;
import ru.yandex.qe.http.aspects.log.LogHttpResponseInterceptor;
import ru.yandex.qe.http.certificates.KeyManagersProvider;
import ru.yandex.qe.http.certificates.MultiX509TrustManager;
import ru.yandex.qe.http.certificates.PemParser;
import ru.yandex.qe.http.certificates.TrustManagersProvider;
import ru.yandex.qe.http.retries.RetryStrategy;
import ru.yandex.qe.logging.security.AuthorizationSecurityGuard;
import ru.yandex.qe.logging.security.CookieSecurityGuard;
import ru.yandex.qe.logging.security.PrivateHeaderSecurityGuard;

/**
 * User: terry Date: 29.06.13 Time: 22:58
 */
public class HttpClientBuilder {

    private final List<HttpAspectFactory> aspectFactories = new LinkedList<>();

    protected RequestConfig.Builder requestConfigBuilder = RequestConfig.custom();
    protected org.apache.http.impl.client.HttpClientBuilder httpClientBuilder = HttpClients.custom();
    protected HttpConnectionManagerBuilder httpConnectionManagerBuilder = new HttpConnectionManagerBuilder();
    protected RetryStrategy httpRetryStrategy;
    protected boolean ignoreCertificates = false;
    protected boolean disableRedirectHandling = false;

    protected HttpRequestExecutor httpRequestExecutor = null;
    protected KeyManagersProvider privateKeyHolder;
    protected List<TrustManager> trustManagers = new ArrayList<>();

    public HttpClientBuilder setIgnoreCertificates(boolean ignoreCertificates) {
        this.ignoreCertificates = ignoreCertificates;
        return this;
    }

    public HttpClientBuilder setDisableRedirectHandling(boolean disableRedirectHandling) {
        this.disableRedirectHandling = disableRedirectHandling;
        return this;
    }

    public HttpClientBuilder setConnectTimeout(int connectTimeoutMs) {
        requestConfigBuilder.setConnectTimeout(connectTimeoutMs);
        this.httpConnectionManagerBuilder.setSocketTimeout(connectTimeoutMs);
        return this;
    }

    public HttpClientBuilder setSocketTimeout(int socketTimeoutMs) {
        requestConfigBuilder.setSocketTimeout(socketTimeoutMs);
        return this;
    }


    public HttpClientBuilder setExpectContinueEnabled(boolean expectContinueEnabled) {
        requestConfigBuilder.setExpectContinueEnabled(expectContinueEnabled);
        return this;
    }


    public HttpClientBuilder setLogEnable(boolean logEnable) {
        return setLogEnable(logEnable, Arrays.asList(new AuthorizationSecurityGuard(), new CookieSecurityGuard()));
    }

    public HttpClientBuilder setLogEnable(boolean logEnable, List<PrivateHeaderSecurityGuard> privateHeaderSecurityGuards) {
        if (logEnable) {
            this.aspectFactories.add(new LogHttpAspectFactory(privateHeaderSecurityGuards));
            this.httpClientBuilder.addInterceptorLast(new LogHttpRequestInterceptor(privateHeaderSecurityGuards));
            this.httpClientBuilder.addInterceptorFirst(new LogHttpResponseInterceptor(privateHeaderSecurityGuards));
        }
        return this;
    }

    public HttpClientBuilder setPooledConnectionTTL(long ttl, TimeUnit ttlTimeUnit) {
        httpConnectionManagerBuilder.setPooledConnectionTTL(ttl, ttlTimeUnit);
        return this;
    }

    public HttpClientBuilder setConnectionPoolSize(int maxTotal) {
        httpConnectionManagerBuilder.setConnectionPoolSize(maxTotal);
        return this;
    }

    public HttpClientBuilder setPerRouteConcurrentLimit(int maxPerRoute) {
        httpConnectionManagerBuilder.setPerRouteConcurrentLimit(maxPerRoute);
        return this;
    }

    public HttpClientBuilder setRetryStrategy(RetryStrategy httpRetryStrategy) {
        this.httpRetryStrategy = httpRetryStrategy;
        return this;
    }

    public HttpClientBuilder setRedirectStrategy(RedirectStrategy redirectStrategy) {
        this.httpClientBuilder.setRedirectStrategy(redirectStrategy);
        return this;
    }

    public HttpClientBuilder addAspectFactory(HttpAspectFactory httpAspectFactory) {
        this.aspectFactories.add(httpAspectFactory);
        return this;
    }

    public HttpClientBuilder addRequestInterceptorFirst(HttpRequestInterceptor httpRequestInterceptor) {
        this.httpClientBuilder.addInterceptorFirst(httpRequestInterceptor);
        return this;
    }


    public HttpClientBuilder addRequestInterceptorLast(HttpRequestInterceptor httpRequestInterceptor) {
        this.httpClientBuilder.addInterceptorLast(httpRequestInterceptor);
        return this;
    }

    public HttpClientBuilder addResponseInterceptorFirst(HttpResponseInterceptor httpResponseInterceptor) {
        this.httpClientBuilder.addInterceptorFirst(httpResponseInterceptor);
        return this;
    }


    public HttpClientBuilder addResponseInterceptorLast(HttpResponseInterceptor httpResponseInterceptor) {
        this.httpClientBuilder.addInterceptorLast(httpResponseInterceptor);
        return this;
    }

    public HttpClientBuilder setHttpRequestExecutor(HttpRequestExecutor httpRequestExecutor) {
        this.httpRequestExecutor = httpRequestExecutor;
        return this;
    }

    public HttpClientBuilder setClientAuthentication(File keyFile, File certFile) {
        privateKeyHolder = new KeyManagersProvider(keyFile, certFile);
        return this;
    }

    /**
     * Adds trusted certificate authority (additionally to standard and YandexInternalCA)
     * @param certificateChain trusted chain in pem format
     * @return
     */
    public HttpClientBuilder addTrustedCertificateAuthority(String certificateChain) {
        try {
            List<X509Certificate> x509Certificates = PemParser.parseX509s(certificateChain);
            final KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
            trustStore.load(null, "".toCharArray());
            for (X509Certificate c : x509Certificates) {
                trustStore.setCertificateEntry(c.getSubjectDN().getName(), c);
            }
            TrustManagerFactory trustManagerFactory =
                    TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(trustStore);

            trustManagers.addAll(Arrays.asList(trustManagerFactory.getTrustManagers()));
            return this;
        } catch (Exception e) {
            Throwables.propagate(e);
            // unreachable line
            return null;
        }
    }

    protected Registry<ConnectionSocketFactory> createSSLConnectionFactoryRegistry() {
        return RegistryBuilder.<ConnectionSocketFactory>create()
                .register("https", getHttpsConnectionSocketFactory())
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .build();
    }

    private SSLConnectionSocketFactory getHttpsConnectionSocketFactory() {
        if (ignoreCertificates) {
            return new SSLConnectionSocketFactory(
                    getSslContext(),
                    SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER
            );
        } else {
            return new QeSSLConnectionSockerFactory(
                    getSslContext(), SSLConnectionSocketFactory.BROWSER_COMPATIBLE_HOSTNAME_VERIFIER
            );
        }
    }

    protected SSLContext getSslContext() {
        try {
            SSLContext context = SSLContext.getInstance("TLS");
            TrustManager[] allTrustManagers = mergeTrustManagers();
            context.init(
                    privateKeyHolder == null ? null : privateKeyHolder.getKeyManagers(),
                    allTrustManagers,
                    null);
            return context;
        } catch (Exception e) {
            throw Throwables.propagate(e);
        }
    }

    private TrustManager[] mergeTrustManagers() {
        TrustManager[] defaultTrustManagers = TrustManagersProvider.getInstance().getTrustManagers();
        List<X509TrustManager> x509TrustManagers = Stream.concat(Arrays.stream(defaultTrustManagers), trustManagers.stream()).filter(X509TrustManager.class::isInstance).map(X509TrustManager.class::cast).collect(Collectors.toList());
        List<TrustManager> otherTrustManagers = Stream.concat(Arrays.stream(defaultTrustManagers), trustManagers.stream()).filter((x -> !X509TrustManager.class.isInstance(x))).collect(Collectors.toList());
        X509TrustManager x509TrustManager = new MultiX509TrustManager(x509TrustManagers);
        List<TrustManager> allTrustManagers = new ArrayList<>(otherTrustManagers);
        allTrustManagers.add(x509TrustManager);
        return allTrustManagers.toArray(new TrustManager[allTrustManagers.size()]);
    }

    public CloseableHttpClient build() {
        RequestConfig requestConfig = requestConfigBuilder.build();

        final HttpClientConnectionManager connectionManager = httpConnectionManagerBuilder
                .setConnectionSocketFactoryRegistry(createSSLConnectionFactoryRegistry()).build();

        org.apache.http.impl.client.HttpClientBuilder builder = httpClientBuilder
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager);

        httpClientBuilder.setRequestExecutor(httpRequestExecutor);

        if (httpRetryStrategy != null) {
            builder.setRetryHandler(httpRetryStrategy).setServiceUnavailableRetryStrategy(httpRetryStrategy);
        } else {
            builder.disableAutomaticRetries();
        }

        if (disableRedirectHandling) {
            builder.disableRedirectHandling();
        }

        CloseableHttpClient httpClient = builder.build();

        for (HttpAspectFactory httpAspectFactory : aspectFactories) {
            httpClient = new AspectCapableHttpClient(httpAspectFactory, httpClient);
        }

        return httpClient;
    }
}
