package ru.yandex.qe.http.aspects.log;

import java.util.concurrent.TimeUnit;

import com.google.common.base.Function;
import com.google.common.base.Stopwatch;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.http.aspects.HttpAspect;

/**
 * Created with IntelliJ IDEA.
 * User: terry
 * Date: 30.06.13
 * Time: 17:56
 */
public class LogHttpAspect implements HttpAspect {
    private static final Logger LOG = LoggerFactory.getLogger(LogHttpAspect.class);
    private final Function<HttpRequest, String> toStringFormatter;

    private Stopwatch stopwatch;

    public LogHttpAspect(Function<HttpRequest, String> toStringFormatter) {
        this.toStringFormatter = toStringFormatter;
    }


    @Override
    public void beforeInvoke(HttpHost httpHost, HttpRequest httpRequest, HttpContext httpContext) {
        stopwatch = Stopwatch.createStarted();
        LOG.debug("http request start: {}", toStringFormatter.apply(httpRequest));
    }

    @Override
    public void afterInvoke(HttpResponse httpResponse, HttpHost httpHost, HttpRequest httpRequest, HttpContext httpContext) {
        LOG.debug("http request ended {} ms. with request: {}, resp status: {}, resp headers: {}",
                stopwatch.elapsed(TimeUnit.MILLISECONDS), toStringFormatter.apply(httpRequest), httpResponse.getStatusLine(),
                httpResponse.getAllHeaders());
    }

    @Override
    public void onFail(Exception exception, HttpHost httpHost, HttpRequest httpRequest, HttpContext httpContext) {
        LOG.warn("http request failed {} ms. error: {}, request: {}",
                stopwatch.elapsed(TimeUnit.MILLISECONDS),
                exception.getClass().getName() + (exception.getMessage() != null ? ": " + exception.getMessage() : "")
                , toStringFormatter.apply(httpRequest));
    }
}
