package ru.yandex.qe.http.handler;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import javax.annotation.concurrent.Immutable;

import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.ResponseHandler;
import org.slf4j.Logger;

import static ru.yandex.qe.util.Functions.asConsumer;
import static ru.yandex.qe.util.Functions.asFunction;
import static ru.yandex.qe.util.Throwables.rethrow;

/**
 * Fluent {@link ResponseHandler} implementation to cover most boilerplate like checking http error codes, logging and forming response object.
 * All objects produced by fluent operations are immutable and thus thread-safe and might be cached to static variables.
 * <p>
 * Typical usage:
 * <pre>
 *  String result = httpClient.execute(new HttpGet(uri), defaultHandler().returnString());
 * </pre>
 * <p>
 * Another example:
 * <pre>
 * List<String> result = httpClient.execute(new HttpGet(uri), newHandler()
 *      .on4xx().doThrow(() -> new IllegalArgumentException())
 *      .onNon2xx().recover(r -> Collections.emptyList())
 *      .returnStringList());
 * </pre>
 *
 * @author rurikk
 */
@Immutable
public abstract class FluentResponseHandler<T> implements ResponseHandler<T> {
    private static final Logger log = org.slf4j.LoggerFactory.getLogger(FluentResponseHandler.class);

    public static FluentResponseHandler<?> newHandler() {
        return EmptyResponseHandler.instance;
    }

    public static FluentResponseHandler<?> defaultHandler() {
        return newHandler()
                .onNon2xx().logStatus()
                .onNon2xx().doThrowStatus();
    }

    public Action<T> onMatch(Predicate<HttpResponse> predicate) {
        return new Action<T>() {
            @Override
            public <S> FluentResponseHandler<S> recover(Function<HttpResponseReader, S> fn) {
                return new ChainingResponseHandler<>(FluentResponseHandler.this,
                        r -> predicate.test(r.getHttpResponse()) ? fn.apply(r) : null);
            }
        };
    }

    public Action<T> onCode(Predicate<Integer> predicate) {
        return onMatch(hr -> predicate.test(hr.getStatusLine().getStatusCode()));
    }

    public Action<T> onCode(int code) {
        return onCode(c -> c == code);
    }

    public Action<T> onCodeRange(int minInclusive, int maxExclusive) {
        return onCode(c -> minInclusive <= c && c < maxExclusive);
    }

    public Action<T> on4xx() {
        return onCodeRange(400, 500);
    }

    public Action<T> on5xx() {
        return onCodeRange(500, 600);
    }

    public Action<T> onNon2xx() {
        return onCode(c -> !(200 <= c && c < 300));
    }

    public <S> ResponseHandler<S> returning(Function<HttpResponseReader, S> fn) {
        return new ChainingResponseHandler<>(this, fn);
    }

    public <S> ResponseHandler<S> processStream(Function<InputStream, S> fn) {
        return returning(r -> r.streamContent(fn));
    }

    public ResponseHandler<?> processStream(Consumer<InputStream> consumer) {
        return processStream(asFunction(consumer));
    }

    public ResponseHandler<String> returnString() {
        return returning(HttpResponseReader::asString);
    }

    public ResponseHandler<List<String>> returnStringList() {
        return returning(HttpResponseReader::asStringList);
    }

    public <S> ResponseHandler<S> returnJson(Class<S> clazz) {
        return returning(r -> r.json(clazz));
    }

    protected abstract T handleImpl(HttpResponseReader httpResponse);

    @SuppressWarnings("unchecked")
    @Override
    public T handleResponse(HttpResponse response) throws IOException {
        return handleImpl(new HttpResponseReader(response));
    }

    public interface Action<T> {
        <S> FluentResponseHandler<S> recover(Function<HttpResponseReader, S> recoveryFunction);

        default FluentResponseHandler<T> peek(Consumer<HttpResponseReader> consumer) {
            return recover(asFunction(consumer));
        }

        default FluentResponseHandler<T> logStatus() {
            return peek(r -> log.warn("{}", r.getStatusLine()));
        }

        default FluentResponseHandler<T> run(Runnable run) {
            return peek(asConsumer(run));
        }

        default FluentResponseHandler<T> doThrowFull(Function<HttpResponseReader, Exception> ex) {
            return recover(asFunction(hr -> rethrow(ex.apply(hr))));
        }

        default FluentResponseHandler<T> doThrow(Function<StatusLine, Exception> ex) {
            return doThrowFull(hr -> ex.apply(hr.getStatusLine()));
        }

        default FluentResponseHandler<T> doThrow(Supplier<Exception> exceptionSupplier) {
            return doThrowFull(asFunction(exceptionSupplier));
        }

        default FluentResponseHandler<T> doThrowStatus() {
            return doThrow(l -> new HttpResponseException(l.getStatusCode(), l.getReasonPhrase()));
        }
    }

}
