package ru.yandex.qe.http.handler;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.apache.commons.io.IOUtils;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.util.EntityUtils;

import ru.yandex.qe.json.JsonUtils;

import static java.nio.charset.StandardCharsets.UTF_8;
import static ru.yandex.qe.util.Throwables.rethrow;

/**
 * @author rurikk
 */
public class HttpResponseReader {
    private final HttpResponse httpResponse;

    public HttpResponseReader(HttpResponse httpResponse) {
        this.httpResponse = httpResponse;
    }

    public HttpResponse getHttpResponse() {
        return httpResponse;
    }

    public StatusLine getStatusLine() {
        return httpResponse.getStatusLine();
    }

    public int getStatusCode() {
        return httpResponse.getStatusLine().getStatusCode();
    }

    public <T> T streamContent(Function<InputStream, T> inputStreamFunction) {
        try (InputStream is = httpResponse.getEntity().getContent()) {
            return inputStreamFunction.apply(is);
        } catch (IOException e) {
            throw rethrow(e);
        }
    }

    public String asString() {
        try {
            return EntityUtils.toString(httpResponse.getEntity());
        } catch (IOException e) {
            throw rethrow(e);
        }
    }

    public List<String> asStringList() {
        try {
            return IOUtils.readLines(httpResponse.getEntity().getContent(), UTF_8);
        } catch (IOException e) {
            throw rethrow(e);
        }
    }

    public byte[] asBytes() {
        try {
            return EntityUtils.toByteArray(httpResponse.getEntity());
        } catch (IOException e) {
            throw rethrow(e);
        }
    }

    public <T> T json(Class<T> clazz) {
        return streamContent(is -> JsonUtils.read(is, clazz));
    }

    public <K, V> Map<K, V> jsonMap(Class<K> keyClass, Class<V> valueClass) {
        return streamContent(is -> JsonUtils.readMap(is, keyClass, valueClass));
    }

    public Map<String, String> jsonStringMap() {
        return jsonMap(String.class, String.class);
    }
}
