package ru.yandex.qe.http.retries;

import java.io.IOException;

import org.apache.http.HttpResponse;
import org.apache.http.protocol.HttpContext;

/**
 * Established by terry
 * on 26.01.16.
 */
public class ExponentialBackoffStrategy implements BackoffStrategy {

    /**
     * Base sleep time (milliseconds) for general exceptions. *
     */
    private static final int SCALE_FACTOR = 300;

    /**
     * Maximum exponential back-off time before retrying a request
     */
    private static final int MAX_BACKOFF_IN_MILLISECONDS = 60 * 1000;

    /**
     * Maximum number of retries before the max backoff will be hit. This is
     * calculated via log_2(MAX_BACKOFF_IN_MILLISECONDS / SCALE_FACTOR)
     * based on the code below.
     */
    private static final int MAX_RETRIES_BEFORE_MAX_BACKOFF = 8;

    private final int scaleFactor;

    public ExponentialBackoffStrategy() {
        this(SCALE_FACTOR);
    }

    public ExponentialBackoffStrategy(int scaleFactor) {
        this.scaleFactor = scaleFactor;
    }

    @Override
    public long retryIntervalMs(HttpResponse response, int executionCount, HttpContext context) {
        return delayBeforeNextRetry(executionCount);
    }

    @Override
    public long retryIntervalMs(IOException response, int executionCount, HttpContext context) {
        return delayBeforeNextRetry(executionCount);
    }

    public final long delayBeforeNextRetry(int executionCount) {
        if (executionCount <= 0) return 0;
        if (executionCount > MAX_RETRIES_BEFORE_MAX_BACKOFF) return MAX_BACKOFF_IN_MILLISECONDS;

        long delay = (1L << executionCount - 1) * scaleFactor;
        delay = Math.min(delay, MAX_BACKOFF_IN_MILLISECONDS);
        return delay;
    }
}