package ru.yandex.qe.json;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;

public class JsonUtils {

    private static final DefaultJsonMapper MAPPER = new DefaultJsonMapper();
    private static final StreamingJsonUtils STREAMING_UTILS = new StreamingJsonUtils(MAPPER);

    public static String write(Object object) {
        try {
            return MAPPER.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static void write(Object object, @Nonnull File file) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(file != null, "file is null");
        try {
            MAPPER.writeValue(file, object);
        } catch (IOException e) {
            throw new IllegalStateException("Error write to file - " + file.getAbsolutePath() + ":" + e.getMessage(), e);
        }
    }

    public static void write(Object object, @Nonnull OutputStream stream) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(stream != null, "stream is invalid");
        try {
            MAPPER.writeValue(stream, object);
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static void write(Object object, @Nonnull Writer stream) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(stream != null, "writer is invalid");
        try {
            MAPPER.writeValue(stream, object);
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static <T> void write(@Nonnull Iterator<T> iterator, @Nonnull Writer writer) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(iterator != null, "iterator is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(writer != null, "writer is null");
        STREAMING_UTILS.writeArray(iterator, writer);
    }

    public static <T> void write(@Nonnull Iterator<T> iterator, @Nonnull OutputStream stream) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(iterator != null, "iterator is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(stream != null, "stream is null");
        STREAMING_UTILS.writeArray(iterator, stream);
    }

    public static <T> void write(@Nonnull Iterator<T> iterator, @Nonnull File file) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(iterator != null, "iterator is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(file != null, "file is null");
        STREAMING_UTILS.writeArray(iterator, file);
    }

    public static <VALUE> VALUE read(String json, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        if (json == null) {
            return null;
        }
        try {
            return MAPPER.readValue(json, valueClass);
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(@Nonnull File file, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(file != null, "file is null");
        try {
            return MAPPER.readValue(file, valueClass);
        } catch (IOException e) {
            throw new IllegalStateException("Error read from file - " + file.getAbsolutePath() + ":" + e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(@Nonnull URL url, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(url != null, "url is invalid");
        try {
            return MAPPER.readValue(url, valueClass);
        } catch (IOException e) {
            throw new IllegalStateException("Error read from url - " + url.toString() + ":" + e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(@Nonnull InputStream stream, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(stream != null, "stream is invalid");
        try {
            return MAPPER.readValue(stream, valueClass);
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(@Nonnull Reader reader, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(reader != null, "reader is invalid");
        try {
            return MAPPER.readValue(reader, valueClass);
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(String json, @Nonnull TypeReference<VALUE> valueTypeReference) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueTypeReference != null, "value type reference is null");
        if (json == null) {
            return null;
        }
        try {
            return (VALUE) MAPPER.readValue(json, valueTypeReference);
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(@Nonnull File file, @Nonnull TypeReference<VALUE> valueTypeReference) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueTypeReference != null, "value type reference is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(file != null, "file is null");
        try {
            return (VALUE) MAPPER.readValue(file, valueTypeReference);
        } catch (IOException e) {
            throw new IllegalStateException("Error read from file - " + file.getAbsolutePath() + ":" + e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(@Nonnull URL url, @Nonnull TypeReference<VALUE> valueTypeReference) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueTypeReference != null, "value type reference is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(url != null, "url is invalid");
        try {
            return (VALUE) MAPPER.readValue(url, valueTypeReference);
        } catch (IOException e) {
            throw new IllegalStateException("Error read from url - " + url.toString() + ":" + e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(@Nonnull InputStream stream, @Nonnull TypeReference<VALUE> valueTypeReference) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueTypeReference != null, "value type reference is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(stream != null, "stream is invalid");
        try {
            return (VALUE) MAPPER.readValue(stream, valueTypeReference);
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static <VALUE> VALUE read(@Nonnull Reader reader, @Nonnull TypeReference<VALUE> valueTypeReference) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueTypeReference != null, "value type reference is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(reader != null, "reader is invalid");
        try {
            return (VALUE) MAPPER.readValue(reader, valueTypeReference);
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    @Nonnull
    public static <VALUE> List<VALUE> readList(String json, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        if (json == null) {
            return Collections.emptyList();
        }
        try {
            List<VALUE> result = MAPPER.readValue(json, MAPPER.getTypeFactory().constructCollectionType(ArrayList.class, valueClass));
            return result != null ? result : Collections.<VALUE>emptyList();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    @Nonnull
    public static <VALUE> List<VALUE> readList(@Nonnull File file, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(file != null, "file is null");
        try {
            List<VALUE> result = MAPPER.readValue(file, MAPPER.getTypeFactory().constructCollectionType(ArrayList.class, valueClass));
            return result != null ? result : Collections.<VALUE>emptyList();
        } catch (IOException e) {
            throw new IllegalStateException("Error read from file - " + file.getAbsolutePath() + ":" + e.getMessage(), e);
        }
    }

    @Nonnull
    public static <VALUE> List<VALUE> readList(@Nonnull URL url, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(url != null, "url is invalid");
        try {
            List<VALUE> result = MAPPER.readValue(url, MAPPER.getTypeFactory().constructCollectionType(ArrayList.class, valueClass));
            return result != null ? result : Collections.<VALUE>emptyList();
        } catch (IOException e) {
            throw new IllegalStateException("Error read from url - " + url.toString() + ":" + e.getMessage(), e);
        }
    }

    @Nonnull
    public static <VALUE> List<VALUE> readList(@Nonnull InputStream stream, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(stream != null, "stream is invalid");
        try {
            List<VALUE> result = MAPPER.readValue(stream, MAPPER.getTypeFactory().constructCollectionType(ArrayList.class, valueClass));
            return result != null ? result : Collections.<VALUE>emptyList();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    @Nonnull
    public static <VALUE> List<VALUE> readList(@Nonnull Reader reader, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(reader != null, "reader is invalid");
        try {
            List<VALUE> result = MAPPER.readValue(reader, MAPPER.getTypeFactory().constructCollectionType(ArrayList.class, valueClass));
            return result != null ? result : Collections.<VALUE>emptyList();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    @Nonnull
    public static <VALUE> Iterator<VALUE> getArrayIterator(@Nonnull Reader reader, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(reader != null, "reader is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        return STREAMING_UTILS.readArray(reader, valueClass);
    }

    @Nonnull
    public static <VALUE> Iterator<VALUE> getArrayIterator(@Nonnull InputStream input, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(input != null, "input is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        return STREAMING_UTILS.readArray(input, valueClass);
    }

    public static <VALUE> void forEachArrayElement(@Nonnull InputStream jsonArrayStream,
                                                   @Nonnull Class<VALUE> valueClass,
                                                   @Nonnull Consumer<VALUE> consumer)
    {
        //noinspection ConstantConditions
        Preconditions.checkArgument(jsonArrayStream != null, "stream is invalid");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(consumer != null, "consumer is null");
        forEachArrayElementInternal(STREAMING_UTILS.readArray(jsonArrayStream, valueClass), consumer);
    }

    public static <VALUE> void forEachArrayElement(@Nonnull Reader jsonArrayReader,
                                                   @Nonnull Class<VALUE> valueClass,
                                                   @Nonnull Consumer<VALUE> consumer)
    {
        //noinspection ConstantConditions
        Preconditions.checkArgument(jsonArrayReader != null, "reader is invalid");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(consumer != null, "consumer is null");
        forEachArrayElementInternal(STREAMING_UTILS.readArray(jsonArrayReader, valueClass), consumer);
    }

    public static <VALUE> void forEachArrayElement(@Nonnull File jsonArrayFile,
                                                   @Nonnull Class<VALUE> valueClass,
                                                   @Nonnull Consumer<VALUE> consumer)
    {
        //noinspection ConstantConditions
        Preconditions.checkArgument(jsonArrayFile != null, "reader is invalid");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(consumer != null, "consumer is null");
        forEachArrayElementInternal(STREAMING_UTILS.readArray(jsonArrayFile, valueClass), consumer);
    }

    private static <VALUE> void forEachArrayElementInternal(@Nonnull Iterator<VALUE> elements, @Nonnull Consumer<VALUE> consumer) {
        while (elements.hasNext()) {
            consumer.accept(elements.next());
        }
    }

    @Nonnull
    public static <KEY, VALUE> Map<KEY, VALUE> readMap(String json, @Nonnull Class<KEY> keyClass, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(keyClass != null, "key class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        if (json == null) {
            return Collections.emptyMap();
        }
        try {
            Map<KEY, VALUE> result = MAPPER.readValue(json, MAPPER.getTypeFactory().constructMapType(LinkedHashMap.class, keyClass, valueClass));
            return result != null ? result : Collections.<KEY, VALUE>emptyMap();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    @Nonnull
    public static <KEY, VALUE> Map<KEY, VALUE> readMap(@Nonnull File file, @Nonnull Class<KEY> keyClass, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(file != null, "file is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(keyClass != null, "key class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        try {
            Map<KEY, VALUE> result = MAPPER.readValue(file, MAPPER.getTypeFactory().constructMapType(LinkedHashMap.class, keyClass, valueClass));
            return result != null ? result : Collections.<KEY, VALUE>emptyMap();
        } catch (IOException e) {
            throw new IllegalStateException("Error read from file - " + file.getAbsolutePath() + ":" + e.getMessage(), e);
        }
    }

    @Nonnull
    public static <KEY, VALUE> Map<KEY, VALUE> readMap(@Nonnull URL url, @Nonnull Class<KEY> keyClass, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(url != null, "url is invalid");
        //noinspection ConstantConditions
        Preconditions.checkArgument(keyClass != null, "key class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        try {
            Map<KEY, VALUE> result = MAPPER.readValue(url, MAPPER.getTypeFactory().constructMapType(LinkedHashMap.class, keyClass, valueClass));
            return result != null ? result : Collections.<KEY, VALUE>emptyMap();
        } catch (IOException e) {
            throw new IllegalStateException("Error read from url - " + url.toString() + ":" + e.getMessage(), e);
        }
    }

    @Nonnull
    public static <KEY, VALUE> Map<KEY, VALUE> readMap(@Nonnull InputStream stream, @Nonnull Class<KEY> keyClass, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(stream != null, "stream is invalid");
        //noinspection ConstantConditions
        Preconditions.checkArgument(keyClass != null, "key class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        try {
            Map<KEY, VALUE> result = MAPPER.readValue(stream, MAPPER.getTypeFactory().constructMapType(LinkedHashMap.class, keyClass, valueClass));
            return result != null ? result : Collections.<KEY, VALUE>emptyMap();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    @Nonnull
    public static <KEY, VALUE> Map<KEY, VALUE> readMap(@Nonnull Reader reader, @Nonnull Class<KEY> keyClass, @Nonnull Class<VALUE> valueClass) {
        //noinspection ConstantConditions
        Preconditions.checkArgument(reader != null, "reader is invalid");
        //noinspection ConstantConditions
        Preconditions.checkArgument(keyClass != null, "key class is null");
        //noinspection ConstantConditions
        Preconditions.checkArgument(valueClass != null, "value class is null");
        try {
            Map<KEY, VALUE> result = MAPPER.readValue(reader, MAPPER.getTypeFactory().constructMapType(LinkedHashMap.class, keyClass, valueClass));
            return result != null ? result : Collections.<KEY, VALUE>emptyMap();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    public static boolean validate(String json) {
        if (json == null) {
            return false;
        }
        try {
            final JsonParser parser = MAPPER.getFactory().createParser(json);
            //noinspection StatementWithEmptyBody
            while (parser.nextToken() != null) {
                // do nothing
            }
            return true;
        } catch (final IOException e) {
            return false;
        }
    }

    static ObjectMapper getActiveMapper() {
        return MAPPER;
    }
}
