package ru.yandex.qe.json;

import java.io.IOException;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;

/**
 * Adds serializers and deserializers for {@link java.net.URI} and {@link java.net.URL}.
 *
 * @author entropia
 */
public final class URIModule extends SimpleModule {
    private static final long serialVersionUID = 1L;

    public URIModule() {
        addDeserializer(URI.class, new URIDeserializer());
        addDeserializer(URL.class, new URLDeserializer());

        addSerializer(URI.class, new URISerializer());
        addSerializer(URL.class, new URLSerializer());
    }
  
    private static final class URISerializer extends JsonSerializer<URI> {
        @Override
        public void serialize(URI value, JsonGenerator jgen, SerializerProvider provider) throws IOException {
           jgen.writeString(value.toASCIIString());
        }
    }

    private static final class URLSerializer extends JsonSerializer<URL> {
        @Override
        public void serialize(URL value, JsonGenerator jgen, SerializerProvider provider) throws IOException {
          jgen.writeString(value.toString());
        }
    }

    private static final class URIDeserializer extends JsonDeserializer<URI> {
        @Override
        public URI deserialize(JsonParser jp, DeserializationContext ctx) throws IOException {
            final String uri = jp.getValueAsString();
            try {
               return uri == null ? null: new URI(uri);
            } catch (URISyntaxException e) {
                throw new JsonParseException("invalid URI syntax: " + uri, jp.getCurrentLocation(), e);
            }
        }
    }

    private static final class URLDeserializer extends JsonDeserializer<URL> {
        @Override
        public URL deserialize(JsonParser jp, DeserializationContext ctx) throws IOException {
            final String url = jp.getValueAsString();
            try {
                return url == null ? null: new URL(url);
            } catch (MalformedURLException e) {
                throw new JsonParseException("invalid URL syntax: " + url, jp.getCurrentLocation(), e);
            }
        }
    }
}
