package ru.yandex.qe.spring;

import com.google.common.collect.Maps;
import com.google.common.util.concurrent.AtomicLongMap;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.PlaceholderConfigurerSupport;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.util.CollectionUtils;
import org.springframework.util.PropertyPlaceholderHelper;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class PropertyResolverBean extends PropertyPlaceholderConfigurer implements ApplicationContextAware {

    private final static Logger LOG = LoggerFactory.getLogger(PropertyResolverBean.class);

    private static final String PROPERTIES_EXTENSION = ".properties";
    private final static String CONTEXT_PROPERTIES_FILE_PREFIX = "context";
    private final static String EXTERNAL_PROPERTIES_PROPERTY_NAME = "ext.properties";
    private final static String EXTERNAL_PROPERTIES_PROPERTY_DEFAULT_VALUE = "/etc/yandex/chef.properties";
    private ApplicationContext applicationContext;

    private Properties mergedProperties;
    private AtomicLongMap<String> propertyToUsingAmountMap = AtomicLongMap.create();
    private List<Pair<URI, Properties>> locationWithPropertiesList;
    private static List<String> overridingProperties = new LinkedList<>();
    private final List<String> additionalPropertiesLocations = new LinkedList<>();
    private final Set<String> secrets = new HashSet<>();

    @Override
    protected Properties mergeProperties() throws IOException {
        if (mergedProperties == null) {
            synchronized (this) {
                if (mergedProperties == null) {
                    mergedProperties = createMergedProperties();
                }
            }
        }
        return mergedProperties;
    }

    private Properties createMergedProperties() throws IOException {
        mergedProperties = new Properties();
        locationWithPropertiesList = new ArrayList<>();

        final String environment = ServerProfileDetector.detectEnvironment(applicationContext);
        final String hostName = InetAddress.getLocalHost().getHostName();
        LOG.info("Detected server environment \"{}\" on host \"{}\".", environment, hostName);

        // default properties
        Properties defaultProperties = super.mergeProperties();
        defaultProperties.put("qe.hostname", hostName);
        defaultProperties.put("qe.server-profile", System.getProperty("qe.server-profile", environment));
        defaultProperties.put("qe.server-environment", System.getProperty("qe.server-environment", environment));
        CollectionUtils.mergePropertiesIntoMap(defaultProperties, mergedProperties);
        locationWithPropertiesList.add(Pair.of((URI) null, defaultProperties));

        for (String location : additionalPropertiesLocations) {
            tryMergeResourceProp(new ClassPathResource(location), mergedProperties, true);
        }

        final Properties contextProps = loadClassPath(environment, hostName, CONTEXT_PROPERTIES_FILE_PREFIX);
        CollectionUtils.mergePropertiesIntoMap(contextProps, mergedProperties);

        if (System.getProperties().containsKey(EXTERNAL_PROPERTIES_PROPERTY_NAME)) {
            final String path = System.getProperty(EXTERNAL_PROPERTIES_PROPERTY_NAME);
            tryMergeResourceProp(new FileSystemResource(path), mergedProperties, false);
        } else {
            tryMergeResourceProp(new FileSystemResource(EXTERNAL_PROPERTIES_PROPERTY_DEFAULT_VALUE), mergedProperties, false);
        }

        for (String location : overridingProperties) {
            tryMergeResourceProp(new ClassPathResource(location), mergedProperties, true);
        }

        setEnvironmentProperty("QLOUD_MONGO_HOSTS", "qloud.service.mongo");
        setEnvironmentProperty("QLOUD_MONGO_PORT", "qloud.service.mongo.port");
        setEnvironmentProperty("QLOUD_ZOOKEEPER_HOSTS", "qloud.service.zookeeper");
        setEnvironmentProperty("QLOUD_ZOOKEEPER_PORT", "qloud.service.zookeeper.port");
        setEnvironmentProperty("QLOUD_HOSTNAME", "qloud.hostname");
        setEnvironmentProperty("QLOUD_HOSTNAME", "qe.hostname");
        setEnvironmentProperty("QLOUD_HTTP_PORT", "jetty.ServerConnector.port");

        for (Map.Entry<String, String> entry : System.getenv().entrySet()) {
            mergedProperties.put("env."+entry.getKey(), entry.getValue());
        }

        final Set<String> keys = new TreeSet<>(mergedProperties.stringPropertyNames());

        final StringBuilder builder = new StringBuilder();
        builder.append("Setting properties: ");

        final PropertyPlaceholderHelper helper = new PropertyPlaceholderHelper(PlaceholderConfigurerSupport.DEFAULT_PLACEHOLDER_PREFIX,
                PlaceholderConfigurerSupport.DEFAULT_PLACEHOLDER_SUFFIX,
                PlaceholderConfigurerSupport.DEFAULT_VALUE_SEPARATOR,
                true);

        for (final String key: keys) {
            if (key.toLowerCase().contains("password") || key.toLowerCase().contains("token") || key.toLowerCase().contains("passphrase")) {
                secrets.add(helper.replacePlaceholders(mergedProperties.getProperty(key), mergedProperties));
            }
        }

        for (final String key : keys) {
            final String rawValue = mergedProperties.getProperty(key);
            final String interpolatedValue = helper.replacePlaceholders(rawValue, mergedProperties);
            if (rawValue.equals(interpolatedValue)) {
                builder.append("\n").append(key).append("=").append(hideSensitiveData(secrets, rawValue));
            } else {
                builder.append("\n").append(key).append("=").append(hideSensitiveData(secrets, rawValue))
                        .append(" -> ").append(hideSensitiveData(secrets, interpolatedValue));
            }
        }
        LOG.debug(builder.toString());

        return mergedProperties;
    }

    public String getSecureProperty(final String key) {
        return hideSensitiveData(secrets, resolvePlaceholder(key, mergedProperties));
    }

    private void setEnvironmentProperty(final String envKey, final String propKey) {
        final String value = System.getenv(envKey);
        if (value != null) {
            mergedProperties.put(propKey, value);
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    public void setAdditionalPropertiesLocations(List<String> locations) {
        this.additionalPropertiesLocations.addAll(locations);
    }

    public static void setOverridingProperties(final List<String> properties) {
        overridingProperties.addAll(properties);
    }

    protected Properties loadClassPath(String environment, String hostName, String contextPropsPrefix) throws IOException {
        final Properties properties = new Properties();

        final String[] resources = {
                String.format("properties/%s%s", contextPropsPrefix, PROPERTIES_EXTENSION),
                String.format("properties/%s/%s%s", environment, contextPropsPrefix, PROPERTIES_EXTENSION),
                String.format("properties/%s-%s%s", contextPropsPrefix, environment, PROPERTIES_EXTENSION),
                String.format("properties/%s/%s/%s%s", environment, hostName, contextPropsPrefix, PROPERTIES_EXTENSION),
                String.format("properties/%s-%s-%s%s", contextPropsPrefix, hostName, environment, PROPERTIES_EXTENSION),
        };
        for (final String resource : resources) {
            tryMergeResourceProp(new ClassPathResource(resource), properties, false);
        }

        return properties;
    }

    protected void tryMergeResourceProp(Resource resource, Properties destinationProps, boolean failIfMissing) throws IOException {
        if (resource.exists()) {
            LOG.info("Loading properties: {}", resource.getDescription());
            final Properties resourceProperties = new Properties();
            resourceProperties.load(resource.getInputStream());
            CollectionUtils.mergePropertiesIntoMap(resourceProperties, destinationProps);
            locationWithPropertiesList.add(Pair.of(resource.getURI(), resourceProperties));
        } else if (failIfMissing) {
            throw new IOException("Resource not found: " + resource.getDescription());
        } else {
            LOG.info("Resource not found: {}", resource.getDescription());
        }
    }

    @Override
    protected String resolvePlaceholder(String placeholder, Properties props) {
        propertyToUsingAmountMap.incrementAndGet(placeholder);
        return super.resolvePlaceholder(placeholder, props);
    }

    public long getPropertyUsingAmount(@Nonnull String propertyName) {
        return propertyToUsingAmountMap.get(propertyName);
    }

    public Map<String, Long> getPropertiesUsageStatistics() {
        final Map<String, Long> usagesMap = propertyToUsingAmountMap.asMap();
        return mergedProperties.keySet().stream()
            .filter(__ -> __ instanceof String)
            .map(String.class::cast)
            .collect(Collectors.toMap(__ -> __, usagesMap::get, (newer, older) -> newer, Maps::newTreeMap));
    }

    @Nullable
    public String getPropertyValue(@Nonnull String propertyName) {
        return super.resolvePlaceholder(propertyName, mergedProperties);
    }

    private String hideSensitiveData(final Set<String> secrets, String propertyValue) {
        String secureValue = propertyValue;
        for (final String secret : secrets) {
            secureValue=secureValue.replaceAll(Pattern.quote(secret), "***");
        }
        return secureValue;
    }

    @Nonnull
    public Collection<String> getPropertyNames() {
        return mergedProperties.stringPropertyNames();
    }

    @Nonnull
    public List<Pair<URI, String>> getPropertySourceUriWithValueList(@Nonnull String propertyName) {
        List<Pair<URI, String>> result = new ArrayList<>(locationWithPropertiesList.size());
        for (Pair<URI, Properties> locationWithProperties : locationWithPropertiesList) {
            if (locationWithProperties.getRight().containsKey(propertyName)) {
                result.add(Pair.of(
                        locationWithProperties.getLeft(),
                        hideSensitiveData(secrets, locationWithProperties.getRight().getProperty(propertyName))
                ));
            }
        }
        return result;
    }
}