package ru.yandex.qe.spring;

import javax.servlet.http.Cookie;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

/**
 * This class holds HTTP request information (in the case that current thread is executing HTTP request).
 * It is responsibility of other modules to fill in the information.
 * @author lvovich
 */
public class RequestIdentity {

    private Cookie sessionCookie;
    private Cookie sslSessionCookie;
    private Map<String, String> headers = new HashMap<>();
    private String remoteAddress;

    private static ThreadLocal<Stack<RequestIdentity>> stack = new ThreadLocal<Stack<RequestIdentity>>() {
        @Override
        protected Stack<RequestIdentity> initialValue() {
            return new Stack<>();
        }
    };

    /**
     * Copy constructor
     * @param other
     */
    private RequestIdentity(final RequestIdentity other) {
        this.sessionCookie = other.sessionCookie;
        this.sslSessionCookie = other.sslSessionCookie;
        this.headers.putAll(other.headers);
        this.remoteAddress = other.remoteAddress;
    }

    public RequestIdentity() {
    }

    /**
     * Resets the state; current thread has no identity after calling this method.
     */
    public static void reset() {
        stack.get().clear();
    }

    /**
     * Make {@code this} current identity. Previous identity, if any, is saved.
     */
    public void enter() {
        stack.get().push(new RequestIdentity(this));
    }

    /**
     * Restore previous identity.
     */
    public static void exit() {
        stack.get().pop();
    }

    /**
     * Get current identity.
     * @return a copy of current identity, It is modifiable, but changes will not affect current thread until you
     * call {@link #enter()} on the modified instance. It there is no identity, returns empty one.
     */
    public static RequestIdentity get() {
        if (stack.get().empty()) {
            return new RequestIdentity();
        } else {
            return new RequestIdentity(stack.get().peek());
        }
    }

    /**
     * Session cookie is a cookie, which enclosing system considers to be identifying request origin.
     * For example, "session_id".
     * @param cookie the cookie to set
     * @return {@code this}
     */
    public void setSessionCookie(Cookie cookie) {
        this.sessionCookie = cookie;
    }

    /**
     * Add header, which somehow identifies the origin of this request, e.g. X-Forwarder-For, Authorization.
     * @param name header name
     * @param value header value IF null, header is removed
     * @return {@code this}
     */
    public void setHeader(final String name, final String value) {
        if (value != null) {
            headers.put(name, value);
        } else {
            headers.remove(name);
        }
    }

    /**
     * The cookie, which identifies this request origin.
     * @return
     */
    public Cookie getSessionCookie() {
        return sessionCookie;
    }

    public Cookie getSslSessionCookie() {
        return sslSessionCookie;
    }

    public void setSslSessionCookie(final Cookie sslSessionCookie) {
        this.sslSessionCookie = sslSessionCookie;
    }

    public String getHeader(final String headerName) {
        return headers.get(headerName);
    }
    /**
     * Remote address of this request
     * @return
     */
    public String getRemoteAddress() {
        return remoteAddress;
    }

    /**
     * Set remote address of this request.
     * @param remoteAddress
     */
    public void setRemoteAddress(final String remoteAddress) {
        this.remoteAddress = remoteAddress;
    }


}
