package ru.yandex.qe.spring;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import ru.yandex.qe.spring.profiles.Profiles;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

/**
 * User: bgleb
 */
public final class ServerProfileDetector {
    private final static Logger LOG = LoggerFactory.getLogger(ServerProfileDetector.class);

    private static final File ENVIRONMENT_MARKER_FILE = new File("/etc/yandex/environment.type");

    public static String detectEnvironment(ApplicationContext applicationContext) {
        String environment = System.getenv("QLOUD_ENVIRONMENT");
        if (environment != null) {
            logEnvironmentSource(environment, "QLOUD_ENVIRONMENT");
            return environment;
        }
        environment = System.getProperty("qe.server-environment", "");
        if (!environment.isEmpty()) {
            logEnvironmentSource(environment, "qe.server-environment");
            return environment;
        }
        // backward compatibility
        final String[] activeProfiles = applicationContext.getEnvironment().getActiveProfiles();
        if (activeProfiles.length > 0) {
            logEnvironmentSource(activeProfiles[0], "first spring profile");
            return activeProfiles[0];
        } else {
            return detectProfile();
        }
    }

    @Deprecated
    public static String detectProfile() {
        String propertyProfile = System.getProperty("qe.server-profile", "");
        if (!propertyProfile.isEmpty()) {
            logEnvironmentSource(propertyProfile, "qe.server-profile");
            return propertyProfile;
        } else if (ENVIRONMENT_MARKER_FILE.exists()) {
            final String env = detectProfileByEnvMarkerFile();
            logEnvironmentSource(env, "/etc/yandex/environment.type");
            return env;
        }
        logEnvironmentSource(Profiles.DEVELOPMENT, "default");
        return Profiles.DEVELOPMENT;
    }

    private static String detectProfileByEnvMarkerFile() {
        try {
            return StringUtils.strip(IOUtils.toString(
                    new FileInputStream(ENVIRONMENT_MARKER_FILE)), " \n\r\t");
        } catch (IOException e) {
            LOG.error("i/o trying get profile from {}", ENVIRONMENT_MARKER_FILE, e);
            throw new RuntimeException("i/o profile detecting", e);
        }
    }

    private static void logEnvironmentSource(String env, String source) {
        LOG.info("Get environment '{}' from {}", env, source);
    }

}
