package ru.yandex.qe.util.units;

import javax.annotation.Nonnull;

/**
 * This class models amount of network bandwidth, in bits per second. Its interface is
 * similar to that of {@code java.time.Duration}.
 *
 * @author entropia
 */
public final class Bandwidth implements Comparable<Bandwidth> {
    /**
     * Represents a zero bandwidth.
     */
    public static final Bandwidth ZERO = new Bandwidth(0L);

    private long bitsPerSecond;

    private Bandwidth(long bitsPerSecond) {
        if (bitsPerSecond < 0) {
            throw new IllegalArgumentException("bandwidth value must be non-negative");
        }
        this.bitsPerSecond = bitsPerSecond;
    }

    @Nonnull
    public static Bandwidth ofBitsPerSecond(long bitsPerSecond) {
        return create(bitsPerSecond);
    }

    @Nonnull
    public static Bandwidth ofKbps(long kilobitsPerSecond) {
        return create(Math.multiplyExact(kilobitsPerSecond, 1000L));
    }

    @Nonnull
    public static Bandwidth ofMbps(long kilobitsPerSecond) {
        return create(Math.multiplyExact(kilobitsPerSecond, 1000000L));
    }

    @Nonnull
    public static Bandwidth ofGbps(long gigabitsPerSecond) {
        return create(Math.multiplyExact(gigabitsPerSecond, 1000000000L));
    }

    @Nonnull
    public static Bandwidth ofBytesPerSecond(long bytesPerSecond) {
        return create(Math.multiplyExact(bytesPerSecond, 8L));
    }

    @Nonnull
    public static Bandwidth ofKBps(long kilobytesPerSecond) {
        return create(Math.multiplyExact(kilobytesPerSecond, 8000L));
    }

    @Nonnull
    public static Bandwidth ofMBps(long megabytesPerSecond) {
        return create(Math.multiplyExact(megabytesPerSecond, 8000000L));
    }

    @Nonnull
    public static Bandwidth ofGBps(long gigabytesPerSecond) {
        return create(Math.multiplyExact(gigabytesPerSecond, 8000000000L));
    }

    @Nonnull
    private static Bandwidth create(long bitsPerSecond) {
        return bitsPerSecond == 0 ? ZERO : new Bandwidth(bitsPerSecond);
    }

    /**
     * @return bandwidth value as bits per second
     */
    public long ofBitsPerSecond() {
        return bitsPerSecond;
    }

    /**
     * @return bandwidth value as kilobits per second
     */
    public long toKbps() {
        return bitsPerSecond / 1000L;
    }

    /**
     * @return bandwidth value as megabits per second
     */
    public long toMbps() {
        return bitsPerSecond / 1000000L;
    }

    /**
     * @return bandwidth value as gigabits per second
     */
    public long toGbps() {
        return bitsPerSecond / 1000000000L;
    }

    /**
     * @return bandwidth value as bytes per second
     */
    public long toBytesPerSecond() {
        return bitsPerSecond / 8L;
    }

    /**
     * @return bandwidth value as kilobytes per second
     */
    public long toKBps() {
        return bitsPerSecond / 8000L;
    }

    /**
     * @return bandwidth value as megabytes per second
     */
    public long toMBps() {
        return bitsPerSecond / 8000000L;
    }

    /**
     * @return bandwidth value as gigabytes per second
     */
    public long toGBps() {
        return bitsPerSecond / 8000000000L;
    }

    /**
     * Returns a copy of this value multiplied by a scalar.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param multiplicand the scalar to multiply this value by; must me {@code >= 0}
     *
     * @return scaled value
     *
     * @see #dividedBy(long)
     * @see #scaledBy(double)
     *
     * @throws IllegalArgumentException if {@code multiplicand} is negative
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Bandwidth multipliedBy(long multiplicand) {
        if (multiplicand < 0) {
            throw new IllegalArgumentException("cannot multiply Bandwidth by a negative scalar");
        }
        return create(Math.multiplyExact(multiplicand, bitsPerSecond));
    }

    /**
     * Returns a copy of this value divided by a scalar.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param divisor the scalar to divide this value by; must be {@code > 0}
     *
     * @return scaled value
     *
     * @see #multipliedBy(long)
     * @see #scaledBy(double)
     *
     * @throws IllegalArgumentException if {@code divisor} is negative
     * @throws ArithmeticException if {@code divisor} is zero
     */
    @Nonnull
    public Bandwidth dividedBy(long divisor) {
        if (divisor < 0) {
            throw new IllegalArgumentException("cannot divide Bandwidth by a negative scalar");
        }
        return create(bitsPerSecond / divisor);
    }

    /**
     * Returns a copy of this value scaled by a floating-point scalar.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param scale the scalar to scale this value by; must be {@code >= 0}
     *
     * @return scaled value
     *
     * @see #multipliedBy(long)
     * @see #dividedBy(long)
     *
     * @throws IllegalArgumentException if {@code scalar} is negative, infinite or not a number
     */
    @Nonnull
    public Bandwidth scaledBy(double scale) {
        if (scale < 0.0) {
            throw new IllegalArgumentException("cannot scale Bandwidth by a negative scalar");
        }
        if (Double.isInfinite(scale)) {
            throw new IllegalArgumentException("cannot scale Bandwidth by an infinite scalar");
        }
        if (Double.isNaN(scale)) {
            throw new IllegalArgumentException("cannot scale Bandwidth by a NaN scalar");
        }

        return create((long) Math.rint(scale * bitsPerSecond));
    }

    /**
     * @return {@code true} if this bandwidth value is zero; {@code false} otherwise
     */
    public boolean isZero() {
        return bitsPerSecond == 0L;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Bandwidth)) return false;

        Bandwidth bandwidth = (Bandwidth) o;
        return bitsPerSecond == bandwidth.bitsPerSecond;
    }

    @Override
    public int hashCode() {
        return (int) (bitsPerSecond ^ (bitsPerSecond >>> 32));
    }

    @Override
    public int compareTo(@Nonnull Bandwidth o) {
        return bitsPerSecond == o.bitsPerSecond ? 0 : (bitsPerSecond < o.bitsPerSecond ? -1 : 1);
    }

    @Override
    public String toString() {
        return String.format("%d bps", bitsPerSecond);
    }
}
