package ru.yandex.qe.util.units;

import javax.annotation.Nonnull;

import ru.yandex.qe.util.HumanReadable;

/**
 * This class models amount of memory or disk space, in bytes. Its interface is
 * similar to that of {@code java.time.Duration}.
 *
 * @author entropia
 */
public final class Space implements Comparable<Space> {
    /**
     * Represents a zero amount of memory or disk space.
     */
    public static final Space ZERO = new Space(0L);

    private static final long BYTES_IN_KILOBYTE = 1L << 10;
    private static final long BYTES_IN_MEGABYTE = 1L << 20;
    private static final long BYTES_IN_GIGABYTE = 1L << 30;
    private static final long BYTES_IN_TERABYTE = 1L << 40;
    private static final long BYTES_IN_PETABYTE = 1L << 50;
    private static final long BYTES_IN_EXABYTE = 1L << 60;

    private final long bytes;

    private Space(long bytes) {
        if (bytes < 0) {
            throw new IllegalArgumentException("size must be non-negative");
        }
        this.bytes = bytes;
    }

    @Nonnull
    public static Space ofBytes(long bytes) {
        return create(bytes);
    }

    @Nonnull
    public static Space ofKilobytes(long kilobytes) {
        return create(Math.multiplyExact(kilobytes, BYTES_IN_KILOBYTE));
    }

    @Nonnull
    public static Space ofMegabytes(long megabytes) {
        return create(Math.multiplyExact(megabytes, BYTES_IN_MEGABYTE));
    }

    @Nonnull
    public static Space ofGigabytes(long gigabytes) {
        return create(Math.multiplyExact(gigabytes, BYTES_IN_GIGABYTE));
    }

    @Nonnull
    public static Space ofTerabytes(long terabytes) {
        return create(Math.multiplyExact(terabytes, BYTES_IN_TERABYTE));
    }

    @Nonnull
    public static Space ofPetabytes(long petabytes) {
        return create(Math.multiplyExact(petabytes, BYTES_IN_PETABYTE));
    }

    @Nonnull
    public static Space ofExabytes(long exabytes) {
        return create(Math.multiplyExact(exabytes, BYTES_IN_EXABYTE));
    }

    @Nonnull
    private static Space create(long bytes) {
        return bytes == 0L ? ZERO : new Space(bytes);
    }

    public long toBytes() {
        return bytes;
    }

    public long toKilobytes() {
        return bytes / BYTES_IN_KILOBYTE;
    }

    public long toMegabytes() {
        return bytes / BYTES_IN_MEGABYTE;
    }

    public long toGigabytes() {
        return bytes / BYTES_IN_GIGABYTE;
    }

    public long toTerabytes() {
        return bytes / BYTES_IN_TERABYTE;
    }

    public long toPetabytes() {
        return bytes / BYTES_IN_PETABYTE;
    }

    public long toExabytes() {
        return bytes / BYTES_IN_EXABYTE;
    }

    /**
     * @return {@code true} if this is a zero amount of memory or disk space;
     *         {@code false} otherwise
     *
     * @see #ZERO
     */
    public boolean isZero() {
        return bytes == 0;
    }

    /**
     * Returns a copy of this value multiplied by a scalar.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param multiplicand the scalar to multiply this value by; must me {@code >= 0}
     *
     * @return scaled value
     *
     * @see #dividedBy(long)
     * @see #scaledBy(double)
     *
     * @throws IllegalArgumentException if {@code multiplicand} is negative
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space multipliedBy(long multiplicand) {
        if (multiplicand < 0) {
            throw new IllegalArgumentException("cannot multiply Space by a negative scalar");
        }
        return create(Math.multiplyExact(multiplicand, bytes));
    }

    /**
     * Returns a copy of this value divided by a scalar.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param divisor the scalar to divide this value by; must be {@code > 0}
     *
     * @return scaled value
     *
     * @see #multipliedBy(long)
     * @see #scaledBy(double)
     *
     * @throws IllegalArgumentException if {@code divisor} is negative
     * @throws ArithmeticException if {@code divisor} is zero
     */
    @Nonnull
    public Space dividedBy(long divisor) {
        if (divisor < 0) {
            throw new IllegalArgumentException("cannot divide Space by a negative scalar");
        }
        return create(bytes / divisor);
    }

    /**
     * Returns a copy of this value scaled by a floating-point scalar.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param scale the scalar to scale this value by; must be {@code >= 0}
     *
     * @return scaled value
     *
     * @see #multipliedBy(long)
     * @see #dividedBy(long)
     *
     * @throws IllegalArgumentException if {@code scalar} is negative, infinite or not a number
     */
    @Nonnull
    public Space scaledBy(double scale) {
        if (scale < 0.0) {
            throw new IllegalArgumentException("cannot scale Space by a negative scalar");
        }
        if (Double.isInfinite(scale)) {
            throw new IllegalArgumentException("cannot scale Space by an infinite scalar");
        }
        if (Double.isNaN(scale)) {
            throw new IllegalArgumentException("cannot scale Space by a NaN scalar");
        }

        return create((long) Math.rint(scale * bytes));
    }

    /**
     * Returns a copy of this value with the specified amount of space added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param space space to add to this value
     *
     * @return a {@code Space} based on this space amount with the specified amount of space added
     *
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space plus(@Nonnull Space space) {
        return plusBytes(space.bytes);
    }

    /**
     * Returns a copy of this value with the specified amount of space subtracted.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param space space to subtract from this value
     *
     * @return a {@code Space} based on this space amount with the specified amount of space subtracted
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if this value is smaller than the amount of space subtracted
     *
     * @see #safeMinus(Space)
     */
    @Nonnull
    public Space minus(@Nonnull Space space) {
        return plusBytes(-space.bytes);
    }

    /**
     * Returns a copy of this value with the specified amount of bytes added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param bytesToAdd the bytes to add; might be negative
     *
     * @return a {@code Space} based on this space amount with the specified bytes added
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if the result will have a negative amount of bytes
     */
    @Nonnull
    public Space plusBytes(long bytesToAdd) {
        if (bytes < -bytesToAdd) {
            throw new IllegalArgumentException("Cannot subtract more than " + bytes + " bytes");
        }

        return create(Math.addExact(bytes, bytesToAdd));
    }

    /**
     * Returns a copy of this value with the specified amount of bytes subtracted.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param bytesToSubtract the bytes to subtract; might be negative
     *
     * @return a {@code Space} based on this space amount with the specified bytes subtracted
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if the result will have a negative amount of bytes
     */
    @Nonnull
    public Space minusBytes(long bytesToSubtract) {
        return plusBytes(-bytesToSubtract);
    }

    /**
     * Returns a copy of this value with the specified amount of kilobytes added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param kilobytesToAdd the kilobytes to add
     *
     * @return a {@code Space} based on this space amount with the specified kilobytes added
     *
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space plusKilobytes(long kilobytesToAdd) {
        return plusBytes(Math.multiplyExact(BYTES_IN_KILOBYTE, kilobytesToAdd));
    }

    /**
     * Returns a copy of this value with the specified amount of kilobytes subtracted.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param kilobytesToSubtract the kilobytes to subtract; might be negative
     *
     * @return a {@code Space} based on this space amount with the specified kilobytes subtracted
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if the result will have a negative amount of bytes
     */
    @Nonnull
    public Space minusKilobytes(long kilobytesToSubtract) {
        return plusKilobytes(-kilobytesToSubtract);
    }

    /**
     * Returns a copy of this value with the specified amount of megabytes added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param megabytesToAdd the megabytes to add
     *
     * @return a {@code Space} based on this space amount with the specified megabytes added
     *
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space plusMegabytes(long megabytesToAdd) {
        return plusBytes(Math.multiplyExact(BYTES_IN_MEGABYTE, megabytesToAdd));
    }

    /**
     * Returns a copy of this value with the specified amount of megabytes subtracted.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param megabytesToSubtract the megabytes to subtract; might be negative
     *
     * @return a {@code Space} based on this space amount with the specified megabytes subtracted
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if the result will have a negative amount of bytes
     */
    @Nonnull
    public Space minusMegabytes(long megabytesToSubtract) {
        return plusMegabytes(-megabytesToSubtract);
    }

    /**
     * Returns a copy of this value with the specified amount of gigabytes added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param gigabytesToAdd the gigabytes to add
     *
     * @return a {@code Space} based on this space amount with the specified gigabytes added
     *
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space plusGigabytes(long gigabytesToAdd) {
        return plusBytes(Math.multiplyExact(BYTES_IN_GIGABYTE, gigabytesToAdd));
    }

    /**
     * Returns a copy of this value with the specified amount of gigabytes subtracted.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param gigabytesToSubtract the gigabytes to subtract; might be negative
     *
     * @return a {@code Space} based on this space amount with the specified gigabytes subtracted
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if the result will have a negative amount of bytes
     */
    @Nonnull
    public Space minusGigabytes(long gigabytesToSubtract) {
        return plusGigabytes(-gigabytesToSubtract);
    }

    /**
     * Returns a copy of this value with the specified amount of terabytes added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param terabytesToAdd the terabytes to add
     *
     * @return a {@code Space} based on this space amount with the specified terabytes added
     *
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space plusTerabytes(long terabytesToAdd) {
        return plusBytes(Math.multiplyExact(BYTES_IN_TERABYTE, terabytesToAdd));
    }

    /**
     * Returns a copy of this value with the specified amount of terabytes subtracted.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param terabytesToSubtract the terabytes to subtract; might be negative
     *
     * @return a {@code Space} based on this space amount with the specified terabytes subtracted
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if the result will have a negative amount of bytes
     */
    @Nonnull
    public Space minusTerabytes(long terabytesToSubtract) {
        return plusTerabytes(-terabytesToSubtract);
    }

    /**
     * Returns a copy of this value with the specified amount of petabytes added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param petabytesToAdd the petabytes to add
     *
     * @return a {@code Space} based on this space amount with the specified petabytes added
     *
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space plusPetabytes(long petabytesToAdd) {
        return plusBytes(Math.multiplyExact(BYTES_IN_PETABYTE, petabytesToAdd));
    }

    /**
     * Returns a copy of this value with the specified amount of petabytes subtracted.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param petabytesToSubtract the petabytes to subtract; might be negative
     *
     * @return a {@code Space} based on this space amount with the specified petabytes subtracted
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if the result will have a negative amount of bytes
     */
    @Nonnull
    public Space minusPetabytes(long petabytesToSubtract) {
        return plusPetabytes(-petabytesToSubtract);
    }

    /**
     * Returns a copy of this value with the specified amount of exabytes added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param exabytesToAdd the exabytes to add
     *
     * @return a {@code Space} based on this space amount with the specified exabytes added
     *
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space plusExabytes(long exabytesToAdd) {
        return plusBytes(Math.multiplyExact(BYTES_IN_EXABYTE, exabytesToAdd));
    }

    /**
     * Returns a copy of this value with the specified amount of exabytes subtracted.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param exabytesToSubtract the exabytes to subtract; might be negative
     *
     * @return a {@code Space} based on this space amount with the specified exabytes subtracted
     *
     * @throws ArithmeticException if numeric overflow occurs
     * @throws IllegalArgumentException if the result will have a negative amount of bytes
     */
    @Nonnull
    public Space minusExabytes(long exabytesToSubtract) {
        return plusExabytes(-exabytesToSubtract);
    }

    /**
     * Returns the <em>difference</em> between two amounts of space as an absolute value.
     *
     * @param other amount of space to calculate difference with
     *
     * @return amount representing the difference between this value and {@code other}
     *
     * @throws ArithmeticException if numeric overflow occurs
     */
    @Nonnull
    public Space difference(@Nonnull Space other) {
        final long min = Math.min(bytes, other.bytes);
        final long max = Math.max(bytes, other.bytes);
        return create(Math.subtractExact(max, min));
    }

    /**
     * Returns the amount of space with that is {@code min(this.toBytes(), other.toBytes())} bytes smaller than
     * this amount. Unlike {@link #minus(Space) this.minus(other)}, this method never fails: it will return
     * {@link #ZERO zero} if {@code other >= this}.
     * <p>
     * This instance is immutable and unaffected by this method call.
     * <p>
     * This method is useful to calculate <em>excess</em> space usage. <em>E.g.</em>, if your job or whatever
     * is over memory limit, the excess usage is zero; otherwise it is the difference between the limit and
     * actual memory usage.
     *
     * @param other amount of space to calculate difference with
     *
     * @return a {@code Space} based on this space amount with the specified amount of space subtracted; or
     *         {@link #ZERO zero} amount of space, if the specified amount of space is greater or equal than
     *         this space amount
     *
     * @see #minus(Space)
     */
    @Nonnull
    public Space safeMinus(@Nonnull Space other) {
        return other.bytes >= this.bytes ? Space.ZERO : minus(other);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Space)) return false;

        Space space = (Space) o;
        return bytes == space.bytes;
    }

    @Override
    public int hashCode() {
        return (int) (bytes ^ (bytes >>> 32));
    }

    @Override
    public int compareTo(@Nonnull Space o) {
        return Long.compare(bytes, o.bytes);
    }

    @Override
    public String toString() {
        if (bytes < BYTES_IN_KILOBYTE) {
            return String.format("%d byte%s", bytes, bytes == 1L ? "" : "s");
        }
        return String.format("%s (%d bytes)", HumanReadable.space(this), bytes);
    }
}
