package ru.yandex.inside.goals;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Optional;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;

/**
 * Main client class. Use it as singleton and get every interface objects from it.
 */
public class GoalsClient {

    private final URI uri;
    private final String token;
    private final GoalsServiceImpl goalsService;
    private final HttpUtil httpUtil;
    private final DepartmentsService departmentsService;
    private final ObjectMapper objectMapper = new ObjectMapper()
            .registerModules(new KotlinModule.Builder().build(), new Jdk8Module(), new JavaTimeModule());
    private final HttpClient httpClient;

    /**
     * @param uri        Goals service URI.
     * @param token      Oauth token allowed to use Goals service.
     * @param httpClient Externally created client if any.
     */
    public GoalsClient(URI uri, String token, Optional<HttpClient> httpClient) {
        this.uri = uri;
        this.token = token;
        goalsService = new GoalsServiceImpl(this);
        departmentsService = new DepartmentsServiceImpl(this);
        this.httpClient = httpClient.orElseGet(HttpClient::createDefault);
        httpUtil = new HttpUtil(this);

        objectMapper.registerModules(new Jdk8Module(), new JodaModule());
    }

    /**
     * @param uri   Goals service URI.
     * @param token Oauth token allowed to use Goals service.
     * @throws URISyntaxException If URI string is malformed.
     */
    public GoalsClient(String uri, String token) throws URISyntaxException {
        this(new URI(uri), token, Optional.<HttpClient>empty());
    }

    /**
     * Get service for goals calls.
     */
    public GoalsService getGoalsService() {
        return goalsService;
    }

    /**
     * Get service for department calls.
     */
    public DepartmentsService getDepartmentsService() {
        return departmentsService;
    }

    public URI getUri() {
        return uri;
    }

    public String getToken() {
        return token;
    }

    HttpUtil getHttpUtil() {
        return httpUtil;
    }

    ObjectMapper getObjectMapper() {
        return objectMapper;
    }

    public HttpClient getHttpClient() {
        return httpClient;
    }
}
