package ru.yandex.inside.goals;

import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectReader;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.utils.URIBuilder;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.inside.goals.model.Goal;
import ru.yandex.inside.goals.model.ListResponse;

class GoalsRequestBuilderImpl implements GoalsService.GoalsRequestBuilder {

    private static final Integer DEFAULT_PAGE_SIZE = 20;
    private final GoalsClient client;

    private final DateTimeFormatter dateFormatter = DateTimeFormat.forPattern("YYYY-mm-DD");
    private final JavaType goalsResultType;

    private Optional<Boolean> isPrivate = Optional.empty();
    private List<Goal.Importance> importances = Collections.emptyList();
    private List<Goal.Status> statuses = Collections.emptyList();
    private Optional<Integer> departmentId = Optional.empty();
    private Optional<String> userId = Optional.empty();
    private Optional<LocalDate> deadline = Optional.empty();
    private List<String> tags = Collections.emptyList();
    private Optional<Integer> pageNum = Optional.empty();
    private Optional<Integer> perPage = Optional.of(DEFAULT_PAGE_SIZE);
    private boolean includeConfidential = false;

    public GoalsRequestBuilderImpl(GoalsClient client) {
        this.client = client;
        goalsResultType = client.getObjectMapper().getTypeFactory()
                .constructParametricType(ListResponse.class, Goal.class);
    }

    @Override
    public GoalsService.GoalsRequestBuilder isPrivate(boolean isPrivate) {
        this.isPrivate = Optional.of(isPrivate);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder importances(Goal.Importance... importances) {
        this.importances = Arrays.asList(importances);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder statuses(Goal.Status... statuses) {
        this.statuses = Arrays.asList(statuses);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder departmentId(int departmentId) {
        this.departmentId = Optional.of(departmentId);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder userId(String userId) {
        this.userId = Optional.of(userId);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder deadline(LocalDate deadline) {
        this.deadline = Optional.of(deadline);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder tags(String... tags) {
        this.tags = Arrays.asList(tags);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder pageNum(int pageNum) {
        this.pageNum = Optional.of(pageNum);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder perPage(int perPage) {
        this.perPage = Optional.of(perPage);
        return this;
    }

    @Override
    public GoalsService.GoalsRequestBuilder withConfidential() {
        includeConfidential = true;
        return this;
    }

    @SuppressWarnings("unchecked")
    @Override
    public ListResponse<Goal> getGoals() throws ClientProtocolException {
        URIBuilder uriBuilder = new URIBuilder(client.getUri()).setPath("/api/v1/goals");

        if (isPrivate.isPresent()) {
            uriBuilder.addParameter("is_private", isPrivate.get().toString());
        }

        if (importances != null && !importances.isEmpty()) {
            String parameter = importances.stream()
                    .map(Enum::ordinal)
                    .map(i -> Integer.toString(i))
                    .collect(Collectors.joining(","));

            uriBuilder.addParameter("importance", parameter);
        }

        if (statuses != null && !statuses.isEmpty()) {
            String parameter = statuses.stream()
                    .map(Enum::ordinal)
                    .map(i -> Integer.toString(i))
                    .collect(Collectors.joining(","));

            uriBuilder.addParameter("status", parameter);
        }

        if (departmentId.isPresent()) {
            uriBuilder.addParameter("department_id", departmentId.get().toString());
        }

        if (userId.isPresent()) {
            uriBuilder.addParameter("person_id", userId.get());
        }

        if (deadline.isPresent()) {
            uriBuilder.addParameter("deadline", dateFormatter.print(deadline.get()));
        }

        if (tags != null && !tags.isEmpty()) {
            uriBuilder.addParameter("tag", String.join(",", tags));
        }

        if (pageNum.isPresent()) {
            Integer pageSize = perPage.orElse(DEFAULT_PAGE_SIZE);
            uriBuilder.addParameter("page", pageNum.get().toString())
                    .addParameter("page_size", pageSize.toString());
        }

        if (includeConfidential) {
            uriBuilder.addParameter("include_confidential", "true");
        }

        ObjectReader reader = client.getObjectMapper().reader(goalsResultType);

        return ((ListResponse<Goal>) client.getHttpUtil().doGetRequest(uriBuilder, reader));
    }

    @Override
    public Iterator<List<Goal>> getPaginatedGoals() {
        return new Iterator<List<Goal>>() {
            private boolean hasNext = true;
            private int pageNum = 1;

            @Override
            public boolean hasNext() {
                return hasNext;
            }

            @Override
            public List<Goal> next() {
                ListResponse<ru.yandex.inside.goals.model.Goal> response;

                try {
                    response = pageNum(pageNum)
                            .getGoals();
                } catch (ClientProtocolException e) {
                    throw new RuntimeException(e);
                }

                pageNum++;
                hasNext = response.getNext().isPresent();

                return response.getResults();
            }
        };
    }
}
