package ru.yandex.inside.goals;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;

import org.apache.http.client.ClientProtocolException;
import org.joda.time.LocalDate;

import ru.yandex.inside.goals.model.Goal;
import ru.yandex.inside.goals.model.GoalLog;
import ru.yandex.inside.goals.model.ListResponse;

/**
 * Interface to goals.
 */
public interface GoalsService {

    /**
     * Build request for fetching goals.
     */
    GoalsRequestBuilder getGoalsBuilder();

    /**
     * Minimal version of goals fetcher. No filters but pagination is on. Intended to be used for downloading
     * everything.
     *
     * @param pageNum Page number starting with 1.
     * @param perPage Goals per page.
     * @return Result object with total record count and links to next/previous pages.
     * @throws ClientProtocolException If anything goes wrong. Including network problems and incorrect response from
     *                                 Goals service. Exception cause will contain original problem.
     */
    ListResponse<Goal> getGoals(int pageNum, int perPage) throws ClientProtocolException;

    /**
     * Fetch goal history.
     *
     * @param id Goal ID
     * @return List of events.
     * @throws ClientProtocolException If anything goes wrong.
     */
    List<GoalLog> getGoalLog(int id) throws ClientProtocolException;

    /**
     * Just parse JSON goal to object.
     *
     * @param json JSON source
     * @return Goal
     * @throws IOException If JSON was not correct.
     */
    Goal parseGoal(String json) throws IOException;

    interface GoalsRequestBuilder {

        /**
         * Return only private goals or only not private.
         */
        GoalsRequestBuilder isPrivate(boolean isPrivate);

        /**
         * Return only goals with specified importances.
         */
        GoalsRequestBuilder importances(Goal.Importance... importances);

        /**
         * Return only goals with specified statuses.
         */
        GoalsRequestBuilder statuses(Goal.Status... statuses);

        /**
         * Return only goals of specified department.
         */
        GoalsRequestBuilder departmentId(int departmentId);

        /**
         * Return only goals of specified user.
         */
        GoalsRequestBuilder userId(String userId);

        /**
         * Return only goals with specified deadline.
         */
        GoalsRequestBuilder deadline(LocalDate deadline);

        /**
         * Return only goals with specified tags.
         */
        GoalsRequestBuilder tags(String... tags);

        /**
         * Page number starting with 1.
         */
        GoalsRequestBuilder pageNum(int pageNum);

        /**
         * Goals per page. If not set and {@code pageNum} is set then 20 is used as default.
         */
        GoalsRequestBuilder perPage(int perPage);

        /**
         * Include confidential goals to response. They will contain only IDs though.
         */
        GoalsRequestBuilder withConfidential();

        /**
         * Perform a request. Builder remains reusable after it.
         *
         * @return Result object with total record count and links to next/previous pages.
         * @throws ClientProtocolException If anything goes wrong. Including network problems and incorrect response from
         *                                 Goals service. Exception cause will contain original problem.
         */
        ListResponse<Goal> getGoals() throws ClientProtocolException;

        /**
         * Iterate through goal pages. Page size is set with {@link #perPage(int)}.
         *
         * @throws RuntimeException If goals cannot be fetched.
         */
        Iterator<List<Goal>> getPaginatedGoals();
    }
}
