package ru.yandex.inside.goals.model;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;

/**
 * Single goal object.
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class Goal {

    public enum Importance {
        NOT_KEY,
        DEPARTMENT,
        COMPANY,
        PRIVATE,
        OKR,
        UNKNOWN
    }

    public enum Status {
        PLANNED,
        RISK,
        BLOCKED,
        CANCELLED,
        REACHED,
        NEW,
        UNKNOWN
    }

    private final Long id;
    private final List<GoalUserRef> implementers;
    private final List<GoalUserRef> customers;
    private final List<String> tags;
    private final Optional<UserRef> responsible;
    private final Optional<String> title;
    private final Optional<String> description;
    private final Optional<String> comment;
    private final Status status;
    private final Importance importance;
    private final Optional<Boolean> confidential;

    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd")
    private Optional<LocalDate> deadline;

    private DateTime resolveDateTime;

    @JsonCreator
    public Goal(@JsonProperty("resolve_datetime") DateTime resolveDateTime,
                @JsonProperty("deadline") Optional<LocalDate> deadline,
                @JsonProperty("importance") Integer importanceId,
                @JsonProperty("status") Integer statusId,
                @JsonProperty("comment") Optional<String> comment,
                @JsonProperty("description") Optional<String> description,
                @JsonProperty("title") Optional<String> title,
                @JsonProperty("responsible") Optional<UserRef> responsible,
                @JsonProperty("tags") List<TagRef> tags,
                @JsonProperty("customers") List<GoalUserRef> customers,
                @JsonProperty("implementers") List<GoalUserRef> implementers,
                @JsonProperty("id") Long id,
                @JsonProperty("is_confidential") Optional<Boolean> confidential) {
        this.resolveDateTime = resolveDateTime;
        this.deadline = deadline;

        this.importance = importanceId == null ? Importance.UNKNOWN : Importance.values()[importanceId];
        this.status = statusId == null ? Status.UNKNOWN : Status.values()[statusId];
        this.comment = comment;
        this.description = description;
        this.title = title;
        this.responsible = responsible;
        this.tags = Optional.ofNullable(tags)
                .orElse(Collections.emptyList())
                .stream()
                .map(TagRef::getText)
                .collect(Collectors.toList());
        this.customers = customers;
        this.implementers = implementers;
        this.id = id;
        this.confidential = confidential;
    }

    public Long getId() {
        return id;
    }

    public List<GoalUserRef> getImplementers() {
        return implementers != null ? implementers : Collections.emptyList();
    }

    public List<GoalUserRef> getCustomers() {
        return customers != null ? customers : Collections.emptyList();
    }

    public List<String> getTags() {
        return tags != null ? tags : Collections.emptyList();
    }

    public Optional<UserRef> getResponsible() {
        return responsible;
    }

    public Optional<String> getTitle() {
        return title;
    }

    public Optional<String> getDescription() {
        return description;
    }

    public Optional<String> getComment() {
        return comment;
    }

    public Status getStatus() {
        return status;
    }

    public Importance getImportance() {
        return importance;
    }

    public Optional<LocalDate> getDeadline() {
        return deadline;
    }

    public DateTime getResolveDateTime() {
        return resolveDateTime;
    }

    public Optional<Boolean> getConfidential() {
        return confidential;
    }
}
