package ru.yandex.qe.hitman.tvm;

import java.util.Collections;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.qe.hitman.tvm.qloud.QloudTvmConfiguration;
import ru.yandex.qe.hitman.tvm.qloud.QloudTvmService;
import ru.yandex.qe.hitman.tvm.qloud.QloudTvmServiceImpl;
import ru.yandex.qe.hitman.tvm.qloud.QloudTvmServiceStub;
import ru.yandex.qe.hitman.tvm.qloud.TvmDaemonConfiguration;

@Configuration
public class QloudTvmSpringConfiguration {

    @Bean
    public TvmDaemonConfiguration tvmDaemonConfiguration(
            @Value("${qloud.tvm.configuration:#{systemEnvironment['QLOUD_TVM_CONFIG']}}") final String rawConfiguration) {
        if (rawConfiguration == null || rawConfiguration.isEmpty()) {
            return new TvmDaemonConfiguration(TvmDaemonConfiguration.BlackboxEnvironmentType.TEST_YA_TEAM, null,
                    Collections.emptyMap());
        } else {
            return QloudTvmServiceImpl.parseDaemonConfiguration(rawConfiguration);
        }
    }

    @Bean
    public QloudTvmConfiguration qloudTvmConfiguration(
            @Value("${qloud.tvm.application.client.id:#{null}}") final Optional<String> applicationClientId,
            @Value("${qloud.tvm.token:#{systemEnvironment['QLOUD_TVM_TOKEN']}}") final String qloudToken,
            final TvmDaemonConfiguration daemonConfiguration) {
        if (daemonConfiguration.getClients().isEmpty()) {
            if (applicationClientId.isPresent()) {
                throw new IllegalArgumentException(
                        "Got explicit application client id but have no valid configuration");
            } else {
                return new QloudTvmConfiguration("fake", -1, "");
            }
        }

        if (qloudToken == null) {
            throw new IllegalArgumentException("Qloud TVM token is missing - invalid configuration");
        }
        final String applicationAlias;
        if (applicationClientId.isPresent()) {
            applicationAlias = applicationClientId.get();
            if (!daemonConfiguration.getClients().containsKey(applicationAlias)) {
                throw new IllegalArgumentException(
                        String.format("Missing TVM client with name '%s' in configuration. Defined clients: %s",
                                applicationAlias, daemonConfiguration.getClients().keySet()));
            }
        } else {
            if (daemonConfiguration.getClients().size() != 1) {
                throw new IllegalStateException(String.format(
                        "No explicit client name defined, but can't determine client implicitly. Defined clients: %s",
                        daemonConfiguration.getClients().keySet()));
            }

            applicationAlias = daemonConfiguration.getClients().keySet().iterator().next();
        }

        final int clientId = daemonConfiguration.getClients().get(applicationAlias).getId();

        return new QloudTvmConfiguration(applicationAlias, clientId, qloudToken);
    }

    @Bean
    public QloudTvmService qloudTvmService(final QloudTvmConfiguration qloudTvmConfiguration,
                                           final TvmDaemonConfiguration tvmDaemonConfiguration) {
        if (tvmDaemonConfiguration.getClients().isEmpty()) {
            return new QloudTvmServiceStub(Collections.emptySet());
        }
        return QloudTvmServiceImpl.createDefault(qloudTvmConfiguration, tvmDaemonConfiguration);
    }


}
