package ru.yandex.qe.hitman.tvm.qloud;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.StandardHttpRequestRetryHandler;
import org.slf4j.Logger;
import ru.yandex.qe.hitman.tvm.TvmConstants;

import static org.slf4j.LoggerFactory.getLogger;

/**
 * Support for local TVM Qloud agent.
 * You can see configuration steps in <a href=https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon/tvm-getting-started/#nastrojjkaokruzhenijavqloud>documentation</a>
 */
public class QloudTvmServiceImpl implements QloudTvmService {

    private static final Logger LOG = getLogger(QloudTvmServiceImpl.class);

    private final HttpClientBuilder httpClientBuilder;
    private final ObjectMapper objectMapper;

    private final String applicationClientRef;
    private final Set<String> allowedTargetClientIds;
    private final String qloudTvmToken;

    private final URI serviceTicketValidationURI;
    private final URI userTicketValidationURI;

    /**
     * @param applicationClientId field "Name" in TVM secret configuration
     * @param qloudTvmToken       value of the environment variable QLOUD_TVM_TOKEN
     */
    public QloudTvmServiceImpl(final HttpClientBuilder httpClientBuilder,
                               final ObjectMapper objectMapper,
                               final String applicationClientId,
                               final Set<String> allowedTargetClientIds,
                               final String qloudTvmToken) {
        this.objectMapper = objectMapper;
        this.applicationClientRef = applicationClientId;
        this.allowedTargetClientIds = allowedTargetClientIds;
        this.qloudTvmToken = qloudTvmToken;
        this.httpClientBuilder = httpClientBuilder;
        this.serviceTicketValidationURI = getServiceTicketCheckURI(applicationClientId);
        this.userTicketValidationURI = getUserTicketCheckURI();
    }

    public static QloudTvmService createDefault(final String applicationClientId,
                                                final Set<String> allowedTargetClientIds,
                                                final String qloudTvmToken) {
        return new QloudTvmServiceImpl(
                getDefaultHttpClientBuilder(),
                new ObjectMapper().registerModules(new KotlinModule.Builder().build(), new Jdk8Module(),
                        new JavaTimeModule()),
                Objects.requireNonNull(applicationClientId),
                allowedTargetClientIds, Objects.requireNonNull(qloudTvmToken)
        );
    }

    public static QloudTvmService createDefault(final QloudTvmConfiguration configuration,
                                                final TvmDaemonConfiguration tvmDaemonConfiguration) {
        final String applicationClientId = configuration.getApplicationClientId();
        if (!tvmDaemonConfiguration.getClients().containsKey(applicationClientId)) {
            throw new IllegalArgumentException(
                    "No client with id " + applicationClientId + " is defined in daemon configuration "
                            + tvmDaemonConfiguration);
        }
        final Set<String> allowedTargetClientIds =
                tvmDaemonConfiguration.getClients().get(applicationClientId).getDestinations().keySet();
        return createDefault(applicationClientId, allowedTargetClientIds, configuration.getQloudToken());
    }

    public static HttpClientBuilder getDefaultHttpClientBuilder() {
        return HttpClientBuilder.create()
                .setDefaultRequestConfig(
                        RequestConfig.custom()
                                .setConnectionRequestTimeout(1_000)
                                .setSocketTimeout(1_000)
                                .setConnectTimeout(1_000)
                                .build())
                .setDefaultSocketConfig(
                        SocketConfig.custom()
                                .setSoTimeout(1_000)
                                .build())
                .setMaxConnTotal(20)
                .setMaxConnPerRoute(20)
                .setRetryHandler(new StandardHttpRequestRetryHandler(3, false));
    }

    private static URI getTicketURI(final String fromClientId, final String toClientId) {
        try {
            return new URIBuilder()
                    .setScheme("http")
                    .setHost("localhost")
                    .setPort(1)
                    .setPath("/tvm/tickets")
                    .addParameter("src", fromClientId)
                    .addParameter("dsts", toClientId)
                    .build();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private static URI getServiceTicketCheckURI(final String destinationClientId) {
        try {
            return new URIBuilder()
                    .setScheme("http")
                    .setHost("localhost")
                    .setPort(1)
                    .setPath("/tvm/checksrv")
                    .addParameter("dst", destinationClientId)
                    .build();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private static URI getUserTicketCheckURI() {
        try {
            return new URIBuilder()
                    .setScheme("http")
                    .setHost("localhost")
                    .setPort(1)
                    .setPath("/tvm/checkusr")
                    .build();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    public static TvmDaemonConfiguration parseDaemonConfiguration(final String rawConfiguration) {
        if (rawConfiguration == null) {
            throw new IllegalArgumentException("TVM daemon configuration not defined");
        }
        LOG.debug("Will parse tvm configuration: {}", rawConfiguration);
        final ObjectMapper objectMapper =
                new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                        .registerModules(new KotlinModule.Builder().build(), new Jdk8Module(), new JavaTimeModule());
        try {
            return objectMapper.readValue(rawConfiguration, TvmDaemonConfiguration.class);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    /**
     * @param targetClientId either destination alias (from TVM secret configuration) or destination TVM client id
     * @return X-Ya-Service-Ticket header value for given destination
     */
    @Override
    public String getTicket(final String targetClientId) {
        if (!allowedTargetClientIds.contains(targetClientId)) {
            throw new IllegalArgumentException(
                    "Attempting to request ticket for disallowed destination " + targetClientId + ", allowed only "
                            + allowedTargetClientIds);
        }

        final URI uri = getTicketURI(applicationClientRef, targetClientId);
        try (CloseableHttpClient httpClient = httpClientBuilder.build()) {
            final HttpGet method = new HttpGet(uri);
            method.setHeader("Authorization", qloudTvmToken);
            try (CloseableHttpResponse response = httpClient.execute(method)) {
                final String contents = new BufferedReader(new InputStreamReader(response.getEntity().getContent()))
                        .lines()
                        .collect(Collectors.joining("\n"));
                if (response.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_OK) {
                    LOG.error("Got {} with body {} from local TVM agent call {}",
                            response.getStatusLine(), contents.replaceAll("\n", ""), uri.toString());
                    throw new IllegalStateException("Got unexpected response with status '" + response.getStatusLine()
                            + "' from local TVM agent");
                }

                final Map<String, TicketContent> result;
                try {
                    result = objectMapper.readValue(contents, new TypeReference<Map<String, TicketContent>>() {
                    });
                } catch (IOException e) {
                    LOG.error("Failed to parse TVM agent response: " + contents.replaceAll("\n", ""), e);
                    throw new UncheckedIOException(e);
                }

                if (result.containsKey(targetClientId)) {
                    return result.get(targetClientId).getTicket();
                }

                final long targetClientIdLong = parseOrZero(targetClientId);
                for (final TicketContent ticketContent : result.values()) {
                    if (targetClientIdLong == ticketContent.tvmId) {
                        return ticketContent.ticket;
                    }
                }

                throw new IllegalArgumentException(
                        "TVM agent response does not contain ticket for destination " + targetClientId);
            }
        } catch (IOException e) {
            LOG.error(String.format("Failed to query for TVM ticket for client \"%s\"", targetClientId), e);
            throw new UncheckedIOException(e);
        }
    }

    private long parseOrZero(final String targetClientId) {
        try {
            return Long.parseLong(targetClientId);
        } catch (final NumberFormatException ignored) {
            return 0L;
        }
    }

    @Override
    public boolean isValidTarget(final String clientId) {
        return allowedTargetClientIds.contains(clientId);
    }

    @Override
    public Optional<TvmServiceTicketInfo> validateServiceTicket(final String serviceTicket) {
        final HttpGet method = new HttpGet(serviceTicketValidationURI);
        method.setHeader("Authorization", qloudTvmToken);
        method.setHeader(TvmConstants.TVM_SERVICE_HEADER_NAME, serviceTicket);

        return validateTicket(method, TvmServiceTicketInfo.class);
    }

    @Override
    public Optional<TvmUserTicketInfo> validateUserTicket(final String userTicket) {
        final HttpGet method = new HttpGet(userTicketValidationURI);
        method.setHeader("Authorization", qloudTvmToken);
        method.setHeader(TvmConstants.TVM_USER_HEADER_NAME, userTicket);

        return validateTicket(method, TvmUserTicketInfo.class);
    }

    private <T> Optional<T> validateTicket(final HttpGet method,
                                           final Class<T> ticketClass) {
        try (CloseableHttpClient httpClient = httpClientBuilder.build()) {
            try (CloseableHttpResponse response = httpClient.execute(method)) {
                final String contents = new BufferedReader(new InputStreamReader(response.getEntity().getContent()))
                        .lines()
                        .collect(Collectors.joining("\n"));
                if (response.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_OK) {
                    LOG.error("Got {} with body {} from local TVM agent call {}",
                            response.getStatusLine(), contents.replaceAll("\n", ""), method.getURI().toString());
                    return Optional.empty();
                } else {
                    try {
                        return Optional.of(objectMapper.readValue(contents, ticketClass));
                    } catch (IOException e) {
                        LOG.error("Failed to parse TVM agent response: " + contents.replaceAll("\n", ""), e);
                        throw new UncheckedIOException(e);
                    }
                }
            }
        } catch (IOException e) {
            LOG.error("Failed to validate TVM ticket", e);
            throw new UncheckedIOException(e);
        }
    }

    public static class TicketContent {

        private final String ticket;
        private final long tvmId;

        @JsonCreator
        public TicketContent(@JsonProperty("ticket") final String ticket,
                             @JsonProperty("tvm_id") final long tvmId) {
            this.ticket = ticket;
            this.tvmId = tvmId;
        }

        public String getTicket() {
            return ticket;
        }

        public long getTvmId() {
            return tvmId;
        }
    }

}
