import logging
import os
import re
import subprocess

MASTER_BRANCH = 'master'
DEVELOP_BRANCH = 'develop'
RELEASE_MARKER = 'release.marker'
POM_FILE = 'pom.xml'
PARENT_MODULE_PATH = '.'
TARGET_FOLDER = 'target'
TICKETS_FILENAME = 'tickets.txt'
FIND_MODULES_EXPRESSION = r'<module>(\S+)</module>'
FIND_CUR_BRANCH_EXPRESSION = r'\*\s+(\S+)'
FIND_ARTIFACTS_EXPRESSION = r'<artifactId>(\S+)</artifactId>'
FIND_TICKETS_EXPRESSION = r'DISPENSER-[0-9]+'
FIND_TAGS_EXPRESSION_TEMPLATE = r'({artifact}-[0-9]+[.[0-9]+]*{branch})\n'

logging.basicConfig(format='%(asctime)s [%(levelname)s] %(message)s', level=logging.DEBUG)


def main():
    logging.info('Start collecting tickets')
    modules = [PARENT_MODULE_PATH]
    modules.extend(get_modules(PARENT_MODULE_PATH))

    current_branch = get_current_branch(PARENT_MODULE_PATH)
    branch_suffix = get_suffix_for_branch(current_branch)
    tickets = collect_tickets(modules, current_branch, branch_suffix)
    write_tickets(tickets)
    logging.info('FInish collecting tickets = %s' % str(tickets))


def get_current_branch(module):
    git_branch = call_subprocess(module, 'git', 'branch')
    current_branch = re.findall(FIND_CUR_BRANCH_EXPRESSION, git_branch)[0]
    return current_branch


def get_suffix_for_branch(cur_branch):
    if cur_branch != MASTER_BRANCH:
        branch_suffix = '-' + os.path.basename(cur_branch)
    else:
        branch_suffix = ''
    return branch_suffix


def get_tags(module, artifact, branch_suffix):
    git_tags = call_subprocess(module, 'git', 'tag', '-l')
    tags = sorted(re.findall(FIND_TAGS_EXPRESSION_TEMPLATE.format(artifact=artifact, branch=branch_suffix), git_tags), reverse=True)
    return tags


def update_tickets(module, tag, current_branch, tickets):
    log = call_subprocess(module, 'git', 'log', tag + '..' + current_branch)
    module_tickets = (set(re.findall(FIND_TICKETS_EXPRESSION, log)))
    logging.info('In module ' + module + ' updated tickets: ' + str(module_tickets))
    tickets.update(module_tickets)


def collect_tickets(modules, current_branch, branch_suffix):
    tickets = set()
    logging.info('Collecting tickets')
    for module in modules:
        logging.info('Processing module %s' % module)
        if os.path.isfile(os.path.join(module, RELEASE_MARKER)):
            artifact = get_artifact(module)
            logging.info('Release marker exists for %s' % artifact)
            tags = get_tags(module, artifact, branch_suffix)
            if len(tags) > 0:
                logging.info('In module ' + module + ' last tag: ' + tags[0])
                update_tickets(module, tags[0], current_branch, tickets)
    return tickets


def write_tickets(tickets):
    tickets_string = '\n'.join(tickets)
    directory = os.path.join(PARENT_MODULE_PATH, TARGET_FOLDER)
    if not os.path.exists(directory):
        os.mkdir(directory)
    file_name = os.path.join(PARENT_MODULE_PATH, TARGET_FOLDER, TICKETS_FILENAME)
    with open(file_name, 'w') as f:
        f.write(tickets_string)


def get_modules(cur_module):
    with open(os.path.join(cur_module, POM_FILE)) as f:
        result = re.findall(FIND_MODULES_EXPRESSION, f.read())
    modules = []
    for module in result:
        module_path = os.path.join(cur_module, module)
        modules.append(module_path)
    return modules


def get_artifact(module_path):
    with open(os.path.join(module_path, POM_FILE)) as f:
        return re.findall(FIND_ARTIFACTS_EXPRESSION, f.read())[0]


def call_subprocess(directory, *arguments):
    (git_query, error) = subprocess.Popen(arguments, cwd=directory, stdout=subprocess.PIPE, stderr=subprocess.PIPE).communicate()
    return (git_query or error).decode('utf-8')


if __name__ == "__main__":
    main()
