package ru.yandex.qe.dispenser.domain;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Suppliers;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import ru.yandex.qe.dispenser.api.v1.DiBotCampaignGroup;
import ru.yandex.qe.dispenser.api.v1.DiCampaignLight;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.index.NormalizedPrimaryKeyBase;

public class BotCampaignGroup extends NormalizedPrimaryKeyBase<String> {

    @NotNull
    private final String botPreOrderIssueKey;
    @NotNull
    private final List<BigOrder> bigOrders;
    @NotNull
    private final List<CampaignForBot> campaigns;
    @NotNull
    private final String name;
    private final boolean active;
    @NotNull
    private final List<CampaignForBot> activeCampaigns;

    private final Supplier<List<CampaignForBot>> activeCampaignWithPredecessorsSupplier = Suppliers.memoize(this::calcActiveWithPredecessors)::get;

    public BotCampaignGroup(@NotNull final String key,
                            @NotNull final String botPreOrderIssueKey,
                            @NotNull final List<BigOrder> bigOrders,
                            @NotNull final List<CampaignForBot> campaigns,
                            @NotNull final String name,
                            final boolean active) {
        super(key);
        this.botPreOrderIssueKey = botPreOrderIssueKey;
        this.bigOrders = bigOrders;
        this.campaigns = campaigns;
        this.name = name;
        this.active = active;
        this.activeCampaigns = campaigns.stream()
                .filter(CampaignForBot::isActive)
                .collect(Collectors.toList());
    }

    public static Builder builder() {
        return new Builder();
    }

    public boolean ifYpInYt() {
        // TODO Awful shortcut...
        return !"order2022".equals(getKey());
    }

    @NotNull
    public String getBotPreOrderIssueKey() {
        return botPreOrderIssueKey;
    }

    @NotNull
    public List<BigOrder> getBigOrders() {
        return bigOrders;
    }

    @NotNull
    public String getName() {
        return name;
    }

    @NotNull
    public List<CampaignForBot> getCampaigns() {
        return campaigns;
    }

    public boolean isActive() {
        return active;
    }

    @NotNull
    public List<CampaignForBot> getActiveCampaigns() {
        return activeCampaigns;
    }

    private List<CampaignForBot> calcActiveWithPredecessors() {
        return getActiveCampaigns()
                .stream()
                .flatMap(ac -> getCampaigns().stream()
                        .filter(c -> !c.getStartDate().isAfter(ac.getStartDate()))
                        .sorted(Comparator.comparing(CampaignForBot::getStartDate))
                )
                .distinct()
                .collect(Collectors.toList());
    }

    public List<CampaignForBot> getActiveWithPredecessors() {
        return activeCampaignWithPredecessorsSupplier.get();
    }

    public DiBotCampaignGroup toView() {
        final List<Long> bigOrdersView = this.bigOrders.stream().map(BigOrder::getId).collect(Collectors.toList());
        final List<DiCampaignLight> campaignsView = this.campaigns.stream().map(CampaignForBot::toView).collect(Collectors.toList());
        final List<DiCampaignLight> activeCampaignsView = getActiveCampaigns().stream().map(CampaignForBot::toView).toList();
        // TODO Remove later
        DiCampaignLight activeCampaignView = getActiveCampaigns().stream()
                .filter(c -> c.getType() == Campaign.Type.DRAFT)
                .max(Comparator.comparing(CampaignForBot::getStartDate))
                .map(CampaignForBot::toView).orElse(null);
        return new DiBotCampaignGroup(getId(), getKey(), name, active, bigOrdersView, botPreOrderIssueKey,
                activeCampaignView, activeCampaignsView, campaignsView);
    }

    public Builder copyBuilder() {
        return new Builder()
                .setId(getId())
                .setKey(getKey())
                .setName(name)
                .setActive(active)
                .setBotPreOrderIssueKey(botPreOrderIssueKey)
                .setBigOrders(bigOrders)
                .setCampaigns(campaigns);
    }

    @Override
    public String toString() {
        return "BotCampaignGroup{" +
                "botPreOrderIssueKey='" + botPreOrderIssueKey + '\'' +
                ", bigOrders=" + bigOrders +
                ", campaigns=" + campaigns +
                ", name='" + name + '\'' +
                ", active=" + active +
                '}';
    }

    public static class Builder {

        @Nullable
        private Long id;
        @Nullable
        private String key;
        @Nullable
        private String botPreOrderIssueKey;
        @NotNull
        private List<BigOrder> bigOrders = new ArrayList<>();
        @Nullable
        private String name;
        @Nullable
        private Boolean active;
        private List<CampaignForBot> campaigns = new ArrayList<>();

        public Builder setId(final Long id) {
            this.id = id;
            return this;
        }

        public Builder setKey(final String key) {
            this.key = key;
            return this;
        }

        public Builder setBotPreOrderIssueKey(final String botPreOrderIssueKey) {
            this.botPreOrderIssueKey = botPreOrderIssueKey;
            return this;
        }

        public Builder setBigOrders(final List<BigOrder> bigOrders) {
            this.bigOrders = bigOrders;
            return this;
        }

        public Builder addBigOrder(final BigOrder bigOrder) {
            this.bigOrders.add(bigOrder);
            return this;
        }

        public Builder setName(final String name) {
            this.name = name;
            return this;
        }

        public Builder setActive(final boolean active) {
            this.active = active;
            return this;
        }

        public Builder addCampaign(final CampaignForBot setting) {
            this.campaigns.add(setting);
            return this;
        }

        public Builder setCampaigns(final List<CampaignForBot> settings) {
            this.campaigns = settings;
            return this;
        }

        public BotCampaignGroup build() {
            Objects.requireNonNull(key, "Key must be non-null");
            Objects.requireNonNull(botPreOrderIssueKey, "Bot pre-order issue key must be non-null");
            Objects.requireNonNull(active, "Active flag must be non-null");
            Objects.requireNonNull(name, "Name must be non-null");
            final BotCampaignGroup result = new BotCampaignGroup(key, botPreOrderIssueKey, bigOrders, campaigns, name, active);
            if (id != null) {
                result.setId(id);
            }
            return result;
        }
    }

}
