package ru.yandex.qe.dispenser.domain;

import java.time.LocalDate;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import ru.yandex.qe.dispenser.api.v1.DiCampaign;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.index.NormalizedPrimaryKeyBase;

/**
 * Кампания сбора {@link QuotaChangeRequest заявок на ресурсы}. Объединяет несколько крупных заказов железа компании.
 *
 * @see ru.yandex.qe.dispenser.domain.bot.BigOrder
 * @see QuotaChangeRequest
 */
public class Campaign extends NormalizedPrimaryKeyBase<String> {
    @NotNull
    private final String name;
    @NotNull
    private final Status status;
    @NotNull
    private final List<BigOrder> bigOrders;
    @NotNull
    private final LocalDate startDate;
    private final boolean requestCreationDisabled;
    @Deprecated
    private final boolean requestModificationDisabledForNonManagers;
    private final boolean allowedRequestModificationWhenClosed;
    private final boolean allowedModificationOnMissingAdditionalFields;
    @Deprecated
    private final boolean forcedAllowingRequestStatusChange;
    private final boolean singleProviderRequestModeEnabled;
    @Deprecated
    private final boolean allowedRequestCreationForProviderAdmin;
    @Deprecated
    private final boolean allowedRequestCreationForCapacityPlanner;
    @NotNull
    private final Type type;
    private final boolean requestModificationDisabled;

    public Campaign(@NotNull final String key,
                    @NotNull final String name,
                    @NotNull final Status status,
                    @NotNull final LocalDate startDate,
                    @NotNull final List<BigOrder> bigOrders,
                    final boolean requestCreationDisabled,
                    @Deprecated final boolean requestModificationDisabledForNonManagers,
                    final boolean allowedRequestModificationWhenClosed,
                    final boolean allowedModificationOnMissingAdditionalFields,
                    @Deprecated final boolean forcedAllowingRequestStatusChange,
                    final boolean singleProviderRequestModeEnabled,
                    @Deprecated final boolean allowedRequestCreationForProviderAdmin,
                    @Deprecated final boolean allowedRequestCreationForCapacityPlanner,
                    @NotNull final Type type,
                    boolean requestModificationDisabled) {
        super(key);
        this.name = name;
        this.status = status;
        this.startDate = startDate;
        this.bigOrders = bigOrders;
        this.requestCreationDisabled = requestCreationDisabled;
        this.requestModificationDisabledForNonManagers = requestModificationDisabledForNonManagers;
        this.allowedRequestModificationWhenClosed = allowedRequestModificationWhenClosed;
        this.allowedModificationOnMissingAdditionalFields = allowedModificationOnMissingAdditionalFields;
        this.forcedAllowingRequestStatusChange = forcedAllowingRequestStatusChange;
        this.singleProviderRequestModeEnabled = singleProviderRequestModeEnabled;
        this.allowedRequestCreationForProviderAdmin = allowedRequestCreationForProviderAdmin;
        this.allowedRequestCreationForCapacityPlanner = allowedRequestCreationForCapacityPlanner;
        this.type = type;
        this.requestModificationDisabled = requestModificationDisabled;
    }

    @NotNull
    public Status getStatus() {
        return status;
    }

    @NotNull
    public List<BigOrder> getBigOrders() {
        return bigOrders;
    }

    @NotNull
    public String getName() {
        return name;
    }

    @NotNull
    public LocalDate getStartDate() {
        return startDate;
    }

    public boolean isRequestCreationDisabled() {
        return requestCreationDisabled;
    }

    @Deprecated
    public boolean isRequestModificationDisabledForNonManagers() {
        return requestModificationDisabledForNonManagers;
    }

    public boolean isAllowedRequestModificationWhenClosed() {
        return allowedRequestModificationWhenClosed;
    }

    public boolean isAllowedModificationOnMissingAdditionalFields() {
        return allowedModificationOnMissingAdditionalFields;
    }

    @Deprecated
    public boolean isForcedAllowingRequestStatusChange() {
        return forcedAllowingRequestStatusChange;
    }

    public boolean isSingleProviderRequestModeEnabled() {
        return singleProviderRequestModeEnabled;
    }

    @Deprecated
    public boolean isAllowedRequestCreationForProviderAdmin() {
        return allowedRequestCreationForProviderAdmin;
    }

    @Deprecated
    public boolean isAllowedRequestCreationForCapacityPlanner() {
        return allowedRequestCreationForCapacityPlanner;
    }

    @NotNull
    public Type getType() {
        return type;
    }

    public boolean isRequestModificationDisabled() {
        return requestModificationDisabled;
    }

    public DiCampaign toView() {
        final List<DiCampaign.BigOrder> bigOrdersView = bigOrders.stream()
                .map(o -> new DiCampaign.BigOrder(o.getId(), o.getDate(), o.getBigOrderId()))
                .collect(Collectors.toList());
        return new DiCampaign(getId(), getKey(), name, status.name(), startDate, bigOrdersView, requestCreationDisabled,
                requestModificationDisabledForNonManagers, allowedRequestModificationWhenClosed,
                allowedModificationOnMissingAdditionalFields, forcedAllowingRequestStatusChange, singleProviderRequestModeEnabled, allowedRequestCreationForProviderAdmin,
                allowedRequestCreationForCapacityPlanner, type.name(), requestModificationDisabled);
    }

    public Campaign withStatus(final Status newStatus) {
        final Campaign result = new Campaign(getKey(), name, newStatus, startDate, bigOrders, requestCreationDisabled,
                requestModificationDisabledForNonManagers, allowedRequestModificationWhenClosed,
                allowedModificationOnMissingAdditionalFields, forcedAllowingRequestStatusChange, singleProviderRequestModeEnabled,
                allowedRequestCreationForProviderAdmin, allowedRequestCreationForCapacityPlanner, type, requestModificationDisabled);
        if (getId() >= 0) {
            result.setId(getId());
        }
        return result;
    }

    public CampaignForBot forBot(final Map<Long, ru.yandex.qe.dispenser.domain.bot.BigOrder> orders) {
        final List<ru.yandex.qe.dispenser.domain.bot.BigOrder> sortedOrders = getBigOrders().stream()
                .map(o -> orders.get(o.getBigOrderId()))
                .sorted(Comparator.comparing(ru.yandex.qe.dispenser.domain.bot.BigOrder::getId))
                .collect(Collectors.toList());
        return new CampaignForBot(getId(), getKey(), sortedOrders, getName(), getStatus(), getStartDate(), getType());
    }

    public Builder copyBuilder() {
        return new Builder()
                .setId(getId())
                .setKey(getKey())
                .setName(name)
                .setStatus(status)
                .setStartDate(startDate)
                .setBigOrders(bigOrders)
                .setRequestCreationDisabled(requestCreationDisabled)
                .setRequestModificationDisabledForNonManagers(requestModificationDisabledForNonManagers)
                .setAllowedRequestModificationWhenClosed(allowedRequestModificationWhenClosed)
                .setAllowedModificationOnMissingAdditionalFields(allowedModificationOnMissingAdditionalFields)
                .setForcedAllowingRequestStatusChange(forcedAllowingRequestStatusChange)
                .setSingleProviderRequestModeEnabled(singleProviderRequestModeEnabled)
                .setAllowedRequestCreationForProviderAdmin(allowedRequestCreationForProviderAdmin)
                .setAllowedRequestCreationForCapacityPlanner(allowedRequestCreationForCapacityPlanner)
                .setRequestModificationDisabled(requestModificationDisabled)
                .setType(type);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder extends CampaignUpdate.BuilderBase<Builder>{
        @Nullable
        private Long id;
        @Nullable
        private String key;
        @Nullable
        private Type type;

        public Builder setId(final long id) {
            this.id = id;
            return this;
        }

        public Builder setKey(final String key) {
            this.key = key;
            return this;
        }

        public Builder setType(Type type) {
            this.type = type;
            return this;
        }

        @Override
        protected Builder self() {
            return this;
        }

        public Campaign build() {
            Objects.requireNonNull(key, "Key must be non-null");
            Objects.requireNonNull(name, "Name must be non-null");
            Objects.requireNonNull(status, "Status must be non-null");
            Objects.requireNonNull(startDate, "Start date must be non-null");
            Objects.requireNonNull(bigOrders, "Big orders must be non-null");
            Objects.requireNonNull(requestCreationDisabled, "'Request creation disabled' flag must be defined");
            Objects.requireNonNull(requestModificationDisabledForNonManagers, "'Request modification disabled' flag must be defined");
            Objects.requireNonNull(allowedRequestModificationWhenClosed, "'Allow edrequest modification when closed' flag must be defined");
            Objects.requireNonNull(allowedModificationOnMissingAdditionalFields,
                    "'Allowed modification on missing additional fields' flag must be defined");
            Objects.requireNonNull(forcedAllowingRequestStatusChange,
                    "'Force allowed request status change' flag must be defined");
            Objects.requireNonNull(singleProviderRequestModeEnabled, "'Single provider mode' flag must be defined");
            Objects.requireNonNull(allowedRequestCreationForProviderAdmin, "'Allow request creation for provider admin' flag must be defined");
            Objects.requireNonNull(allowedRequestCreationForCapacityPlanner,
                    "'Allow request creation for capacity planner' flag must be defined");
            Objects.requireNonNull(type, "Type must be non-null");
            Objects.requireNonNull(requestModificationDisabled, "'Request modification disabled' flag must be defined");
            final Campaign result = new Campaign(key, name, status, startDate, bigOrders, requestCreationDisabled,
                    requestModificationDisabledForNonManagers, allowedRequestModificationWhenClosed,
                    allowedModificationOnMissingAdditionalFields, forcedAllowingRequestStatusChange, singleProviderRequestModeEnabled, allowedRequestCreationForProviderAdmin,
                    allowedRequestCreationForCapacityPlanner, type, requestModificationDisabled);
            if (id != null) {
                result.setId(id);
            }
            return result;
        }

    }

    public enum Status {
        ACTIVE,
        CLOSED,
    }

    public enum Type {
        DRAFT,
        AGGREGATED,
    }

    public static class BigOrder extends LongIndexBase {
        private final Long bigOrderId;
        private final LocalDate date;

        public BigOrder(final Long bigOrderId, final LocalDate date) {
            this.bigOrderId = bigOrderId;
            this.date = date;
        }

        public BigOrder(final ru.yandex.qe.dispenser.domain.bot.BigOrder bigOrder) {
            this.bigOrderId = bigOrder.getId();
            this.date = bigOrder.getDate();
        }

        public Long getBigOrderId() {
            return bigOrderId;
        }

        public LocalDate getDate() {
            return date;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final BigOrder bigOrder = (BigOrder) o;
            return Objects.equals(bigOrderId, bigOrder.bigOrderId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(bigOrderId);
        }

    }

    public static class CampaignOrder {

        private final long id;
        private final long campaignId;
        private final long bigOrderId;

        public CampaignOrder(final long id, final long campaignId, final long bigOrderId) {
            this.id = id;
            this.campaignId = campaignId;
            this.bigOrderId = bigOrderId;
        }

        public long getId() {
            return id;
        }

        public long getCampaignId() {
            return campaignId;
        }

        public long getBigOrderId() {
            return bigOrderId;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final CampaignOrder that = (CampaignOrder) o;
            return id == that.id;
        }

        @Override
        public int hashCode() {
            return Objects.hash(id);
        }

        @Override
        public String toString() {
            return "CampaignOrder{"
                    + "id=" + id
                    + ", campaignId=" + campaignId
                    + ", bigOrderId=" + bigOrderId
                    + '}';
        }

    }

    public static class CampaignIdOrderId {

        private final long campaignId;
        private final long bigOrderId;

        public CampaignIdOrderId(final long campaignId, final long bigOrderId) {
            this.campaignId = campaignId;
            this.bigOrderId = bigOrderId;
        }

        public long getCampaignId() {
            return campaignId;
        }

        public long getBigOrderId() {
            return bigOrderId;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final CampaignIdOrderId that = (CampaignIdOrderId) o;
            return campaignId == that.campaignId
                    && bigOrderId == that.bigOrderId;
        }

        @Override
        public int hashCode() {
            return Objects.hash(campaignId, bigOrderId);
        }

        @Override
        public String toString() {
            return "CampaignIdOrderId{"
                    + "campaignId=" + campaignId
                    + ", bigOrderId=" + bigOrderId
                    + '}';
        }

    }

}
