package ru.yandex.qe.dispenser.domain;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.DiCampaignLight;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.index.LongIndexable;

public class CampaignForBot implements LongIndexable {

    private final long id;
    @NotNull
    private final String key;
    @NotNull
    private final List<BigOrder> bigOrders;
    @NotNull
    private final String name;
    @NotNull
    private final Campaign.Status status;
    @NotNull
    private final LocalDate startDate;
    @NotNull
    private final Campaign.Type type;

    public CampaignForBot(final long id, @NotNull final String key,
                          @NotNull final List<BigOrder> bigOrders, @NotNull final String name, @NotNull final Campaign.Status status,
                          final @NotNull LocalDate startDate, @NotNull final Campaign.Type type) {
        this.id = id;
        this.key = key;
        this.bigOrders = bigOrders;
        this.name = name;
        this.status = status;
        this.startDate = startDate;
        this.type = type;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public long getId() {
        return id;
    }

    @NotNull
    public String getKey() {
        return key;
    }

    @NotNull
    public List<BigOrder> getBigOrders() {
        return bigOrders;
    }

    @Nullable
    public String getName() {
        return name;
    }

    @NotNull
    public LocalDate getStartDate() {
        return startDate;
    }

    @NotNull
    public Campaign.Status getStatus() {
        return status;
    }

    public boolean isActive() {
        return status == Campaign.Status.ACTIVE;
    }

    @NotNull
    public Campaign.Type getType() {
        return type;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final CampaignForBot that = (CampaignForBot) o;
        return id == that.id;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id);
    }

    public DiCampaignLight toView() {
        return new DiCampaignLight(id, key, name, status.name(), bigOrders.stream().map(BigOrder::getId).sorted().collect(Collectors.toList()), type.name());
    }

    @Override
    public String toString() {
        return "CampaignForBot{" +
                "id=" + id +
                ", key='" + key + '\'' +
                ", bigOrders=" + bigOrders +
                ", name='" + name + '\'' +
                ", status=" + status +
                ", startDate=" + startDate +
                ", type=" + type +
                '}';
    }

    public static class Builder {

        @Nullable
        private Long id;
        @Nullable
        private String key;
        @NotNull
        private List<BigOrder> bigOrders = new ArrayList<>();
        @Nullable
        private String name;
        @Nullable
        private Campaign.Status status;
        @Nullable
        private LocalDate startDate;
        @Nullable
        private Campaign.Type type;

        public Builder setId(final Long id) {
            this.id = id;
            return this;
        }

        public Builder setKey(final String key) {
            this.key = key;
            return this;
        }

        public Builder setBigOrders(final List<BigOrder> bigOrders) {
            this.bigOrders = bigOrders;
            return this;
        }

        public Builder addBigOrder(final BigOrder simpleBigOrder) {
            this.bigOrders.add(simpleBigOrder);
            return this;
        }

        public Builder setName(final String name) {
            this.name = name;
            return this;
        }

        public Builder setStatus(final Campaign.Status status) {
            this.status = status;
            return this;
        }

        public Builder setStartDate(final LocalDate startDate) {
            this.startDate = startDate;
            return this;
        }

        public Builder setType(Campaign.Type type) {
            this.type = type;
            return this;
        }

        public CampaignForBot build() {
            Objects.requireNonNull(id, "Id must be non-null");
            Objects.requireNonNull(key, "Key must be non-null");
            Objects.requireNonNull(bigOrders, "Big orders must be non-null");
            Objects.requireNonNull(status, "Status must be non-null");
            Objects.requireNonNull(startDate, "Start date must be non-null");
            Objects.requireNonNull(name, "Name must be non-null");
            Objects.requireNonNull(type, "Type must be non-null");
            return new CampaignForBot(id, key, bigOrders, name, status, startDate, type);
        }

    }

}
