package ru.yandex.qe.dispenser.domain;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;

@SuppressWarnings("UnstableApiUsage")
@Component
public class ClosestResponsibleHelper {
    private static final Logger LOG = LoggerFactory.getLogger(ClosestResponsibleHelper.class);

    private final HierarchySupplier hierarchySupplier;

    private final LoadingCache<Project, Optional<String>> closestResponsibleCache = CacheBuilder.newBuilder()
            .maximumSize(10_000)
            .expireAfterWrite(1, TimeUnit.MINUTES)
            .build(new CacheLoader<Project, Optional<String>>() {
                @Override
                public Optional<String> load(@NotNull final Project project) throws Exception {
                    final Hierarchy hierarchy = hierarchySupplier.get();
                    return getTopLevelProject(hierarchy, project).map(pair -> selectAssignee(pair.getRight()));
                }
            });

    public ClosestResponsibleHelper(final HierarchySupplier hierarchySupplier) {
        this.hierarchySupplier = hierarchySupplier;
    }

    @NotNull
    public static Optional<Pair<Project, ArrayList<Person>>> getTopLevelProject(final Hierarchy hierarchy, final Project sourceProject) {
        final List<Project> pathToRoot = sourceProject.getPathToRoot();
        for (final Project project : pathToRoot) {
            final Set<Person> responsiblePersons = hierarchy.getProjectReader().getLinkedResponsibles(project);
            if (!responsiblePersons.isEmpty()) {
                return Optional.of(Pair.of(project, new ArrayList<>(responsiblePersons)));
            }
        }
        return Optional.empty();
    }

    @Nullable
    public static String selectAssignee(final List<Person> responsible) {
        return responsible.stream().map(Person::getLogin).sorted().findFirst().orElse(null);
    }

    public Optional<String> getClosestResponsible(final Project project) {
        try {
            return closestResponsibleCache.get(project);
        } catch (ExecutionException e) {
            LOG.error("Cache load error", e);
            return Optional.empty();
        }
    }
}
