package ru.yandex.qe.dispenser.domain;

import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import com.google.common.collect.Sets;
import org.apache.commons.lang3.builder.CompareToBuilder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiMetaField;
import ru.yandex.qe.dispenser.api.v1.DiMetaValueSet;
import ru.yandex.qe.dispenser.api.v1.request.DiEntity;
import ru.yandex.qe.dispenser.api.v1.request.DiResourceAmount;
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.ResourceSegmentationReader;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentReader;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.distributed.Identifier;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.index.NormalizedPrimaryKeyBase;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;

/**
 * Сущность, занимающая {@link Resource ресурсы}. Используется только для провайдера Nirvana.
 * <p>Может представлять собой файл, процесс и т.п., который будет занимать сразу несколько ресурсов.
 * <p>В зависимости от спецификации ({@link EntitySpec#isExpirable()}), может иметь или не иметь срок жизни.
 *
 * @see EntitySpec
 */
public final class Entity extends NormalizedPrimaryKeyBase<Entity.Key> {
    @NotNull
    private final Map<ResourceKey, Long> dimensions;
    @Nullable
    private final Long creationTime;
    @Nullable
    private final Long expirationTime;
    @Nullable
    private final Identifier identifier;


    public Entity(@NotNull final Builder builder) {
        super(new Key(builder.key, ValidationUtils.requireNonNull(builder.spec, "Entity specification required!")));
        if (builder.id != null) {
            setId(builder.id);
        }
        dimensions = builder.dimensions;
        final Set<Resource> dimensionsResources = dimensions.keySet().stream()
                .map(ResourceKey::getResource)
                .collect(Collectors.toSet());
        if (!getKey().getSpec().getResources().containsAll(dimensionsResources)) {
            throw new IllegalStateException("Expected dimensions: '" + getKey().getSpec().getResources() + "', but actual: '" + dimensions.keySet() + "'");
        }
        creationTime = builder.creationTime;
        expirationTime = builder.expirationTime;
        if (!getKey().getSpec().isExpirable() && expirationTime != null) {
            throw new IllegalArgumentException("Can't set TTL for not expirable entity by '" + getKey().getSpec() + "' specification!");
        }
        identifier = builder.identifier;
    }

    @NotNull
    public static Builder builder(@NotNull final String key) {
        return new Builder(key);
    }

    @NotNull
    public static Builder builder(@NotNull final Key key) {
        return new Builder(key.getPublicKey()).spec(key.getSpec());
    }

    @NotNull
    public static Builder builder(@NotNull final Entity entity) {
        final Builder builder = builder(entity.getKey());
        if (entity.hasCreationTime()) {
            builder.creationTime(ValidationUtils.requireNonNull(entity.getCreationTime(), "Creation time is required"));
        }
        Optional.ofNullable(entity.getExpirationTime()).ifPresent(builder::expirationTime);
        ResourceKey.getKeysForEntitySpec(entity.getSpec())
                .forEach(key -> builder.dimension(key, entity.getSize(key)));

        builder.identifier(entity.identifier);
        return builder;
    }

    @NotNull
    public EntitySpec getSpec() {
        return getKey().getSpec();
    }

    @Deprecated
    public long getSize(@NotNull final Resource resource) {
        return dimensions.getOrDefault(ResourceKey.of(resource), 0L);
    }

    public long getSize(@NotNull final ResourceKey resourceKey) {
        return dimensions.getOrDefault(resourceKey, 0L);
    }

    @Nullable
    public Long getCreationTime() {
        return creationTime;
    }

    @Nullable
    public Long getExpirationTime() {
        return expirationTime;
    }

    @Nullable
    public Identifier getIdentifier() {
        return this.identifier;
    }

    public boolean hasCreationTime() {
        return creationTime != null;
    }

    @NotNull
    public DiEntity toView() {
        return toView(false);
    }

    @NotNull
    public DiEntity toView(final boolean extendedMeta) {
        final List<DiResourceAmount> resourceAmounts = CollectionUtils.map(dimensions, (resourceKey, value) -> {
            return DiResourceAmount.ofResource(resourceKey.getResource().getKey().getPublicKey())
                    .withAmount(DiAmount.of(value, resourceKey.getResource().getType().getBaseUnit()))
                    .withSegments(SegmentUtils.getNonAggregationSegmentKeys(resourceKey.getSegments()))
                    .build();
        }).collect(Collectors.toList());
        return DiEntity.withKey(getKey().getPublicKey())
                .bySpecification(getSpec().getKey().getPublicKey())
                .occupies(resourceAmounts)
                .meta(
                        extendedMeta ?
                                DiMetaValueSet.builder()
                                        .set(
                                                DiMetaField.of("identifier", DiMetaField.Type.STRING),
                                                identifier.toString()
                                        )
                                        .set(
                                                DiMetaField.of("creationTime", DiMetaField.Type.LONG),
                                                getCreationTime()
                                        )
                                        .set(
                                                DiMetaField.of("serviceKey", DiMetaField.Type.STRING),
                                                getSpec().getService().getKey()
                                        )
                                        .build() : DiMetaValueSet.EMPTY)
                .build();
    }

    @NotNull
    @Override
    public String toString() {
        return "Entity{key='" + getKey().getPublicKey() + "', dimensions=" + dimensions + "}";
    }

    public static final class Key implements Comparable<Key> {
        @NotNull
        private final String key;
        @NotNull
        private final EntitySpec spec;

        private final int hash;

        public Key(@NotNull final String key, @NotNull final EntitySpec spec) {
            this.key = key;
            this.spec = spec;
            this.hash = 31 * key.hashCode() + spec.hashCode();
            ;
        }

        @NotNull
        public String getPublicKey() {
            return key;
        }

        @NotNull
        public EntitySpec getSpec() {
            return spec;
        }

        @Override
        public int compareTo(@NotNull final Key key) {
            return new CompareToBuilder()
                    .append(getSpec(), key.getSpec())
                    .append(getPublicKey(), key.getPublicKey())
                    .build();
        }

        @Override
        public boolean equals(@Nullable final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            return key.equals(((Key) o).key) && spec.equals(((Key) o).spec);
        }

        @Override
        public int hashCode() {
            return hash;
        }

        @NotNull
        @Override
        public String toString() {
            return "Key{key='" + key + "', entitySpec.key=" + spec.getKey().getPublicKey() + "}";
        }
    }

    public static class Builder implements DtoBuilder<Entity> {
        @Nullable
        private Long id;
        @NotNull
        private final String key;
        @Nullable
        private EntitySpec spec;
        @NotNull
        private final Map<ResourceKey, Long> dimensions = new HashMap<>();
        @Nullable
        private Long creationTime;
        @Nullable
        private Long expirationTime;
        @Nullable
        private Identifier identifier;

        public Builder(@NotNull final String key) {
            this.key = key;
        }

        @NotNull
        public Builder id(final long id) {
            this.id = id;
            return this;
        }

        @NotNull
        public Builder identifier(@Nullable final Identifier identifier) {
            this.identifier = Identifier.
                    cached(identifier);
            return this;
        }

        @NotNull
        public Builder spec(@NotNull final EntitySpec spec) {
            this.spec = spec;
            return this;
        }

        @NotNull
        @Deprecated
        public Builder dimension(@NotNull final Resource resource, final long size) {
            return dimension(resource, DiAmount.of(size, resource.getType().getBaseUnit()));
        }

        @NotNull
        public Builder dimension(@NotNull final ResourceKey resourceKey, final long size) {
            dimensions.put(resourceKey, size);
            return this;
        }

        @NotNull
        @Deprecated
        public Builder dimension(@NotNull final Resource resource, @NotNull final DiAmount size) {
            dimensions.put(ResourceKey.of(resource), resource.getType().getBaseUnit().convert(size));
            return this;
        }

        @NotNull
        public Builder dimension(@NotNull final ResourceKey resourceKey, @NotNull final DiAmount size) {
            dimensions.put(resourceKey, resourceKey.getResource().getType().getBaseUnit().convert(size));
            return this;
        }

        @NotNull
        public Builder creationTime(final long creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        @Nonnull
        public Builder expirationTime(final long expirationTime) {
            this.expirationTime = expirationTime;
            return this;
        }

        @Nonnull
        public Builder ttlIfPresent(@Nullable final Duration ttl) {
            if (ttl == null) {
                return this;
            }
            return expirationTime(Instant.now().plus(ttl).toEpochMilli());
        }

        @NotNull
        @Override
        public Entity build() {
            return new Entity(this);
        }
    }

    public static class ResourceKey {
        private final Resource resource;
        private final Set<Segment> segment;

        public static Stream<ResourceKey> getKeysForEntitySpec(final EntitySpec entitySpec) {
            final ResourceSegmentationReader resourceSegmentationReader = Hierarchy.get().getResourceSegmentationReader();
            final SegmentReader segmentReader = Hierarchy.get().getSegmentReader();

            return entitySpec.getResources().stream().flatMap(r -> {
                final Set<Segmentation> segmentations = resourceSegmentationReader.getSegmentations(r);
                if (segmentations.isEmpty()) {
                    return Stream.of(new ResourceKey(r, Collections.emptySet()));
                }

                final List<Set<Segment>> allSegmentsSet = segmentations.stream()
                        .map(segmentReader::get)
                        .collect(Collectors.toList());

                return Sets.cartesianProduct(allSegmentsSet).stream()
                        .map(sets -> new ResourceKey(r, Sets.newHashSet(sets)));
            });
        }

        public static ResourceKey of(final Resource resource) {
            return new ResourceKey(resource, Collections.emptySet());
        }

        public static ResourceKey of(final Resource resource, final Set<Segment> segment) {
            return new ResourceKey(resource, segment);
        }

        public ResourceKey(final Resource resource, final Set<Segment> segment) {
            this.resource = resource;
            this.segment = segment;
        }

        public Resource getResource() {
            return resource;
        }

        public Set<Segment> getSegments() {
            return segment;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final ResourceKey that = (ResourceKey) o;
            return Objects.equals(resource, that.resource) &&
                    Objects.equals(segment, that.segment);
        }

        @Override
        public int hashCode() {
            return Objects.hash(resource, segment);
        }
    }


}
