package ru.yandex.qe.dispenser.domain;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;

public class MultiProjectFieldsContext implements ProjectFieldsContext {
    private boolean projectsMembersCachesInitialized = false;
    private final Map<String, Set<String>> allProjectCachedMembersByKey = new HashMap<>();
    private final Multimap<Project, String> projectCachedMembersByKey = ArrayListMultimap.create();
    @Nullable
    private final List<Campaign> activeCampaigns;
    @NotNull
    private final Supplier<SingleProjectFieldsContext> forked;

    public MultiProjectFieldsContext(@Nullable final List<Campaign> activeCampaigns) {
        this.activeCampaigns = activeCampaigns;
        this.forked = Suppliers.memoize(() -> new SingleProjectFieldsContext(activeCampaigns));
    }

    public void initializeProjectMembersCachesIfNot() {
        if (!projectsMembersCachesInitialized) {
            projectCachedMembersByKey.putAll(Hierarchy.get()
                    .getProjectReader()
                    .getAllMembers(Hierarchy.get()
                            .getProjectReader()
                            .getAll()
                    ));

            final Project rootProject = Hierarchy
                    .get()
                    .getProjectReader()
                    .getRoot();
            final Set<String> rootProjectPersons = new HashSet<>(projectCachedMembersByKey.get(rootProject));
            allProjectCachedMembersByKey.put(rootProject.getPublicKey(), rootProjectPersons);

            projectsMembersCachesInitialized = true;
        }
    }

    @NotNull
    @Override
    public Set<String> getAllMembersOfProject(@NotNull final Project project) {
        initializeProjectMembersCachesIfNot();
        if (!allProjectCachedMembersByKey.containsKey(project.getPublicKey())) {
            final LinkedList<Project> nonLoadedProjectsByOrder = new LinkedList<>();
            Project leastCachedAncestor = project;

            while (!allProjectCachedMembersByKey.containsKey(leastCachedAncestor.getPublicKey())) {
                nonLoadedProjectsByOrder.addFirst(leastCachedAncestor);
                leastCachedAncestor = leastCachedAncestor.getParent();
            }

            final Set<String> cachedProjectMembers = new HashSet<>(allProjectCachedMembersByKey.get(leastCachedAncestor.getPublicKey()));

            for (final Project nonCachedProject : nonLoadedProjectsByOrder) {
                cachedProjectMembers.addAll(projectCachedMembersByKey.get(nonCachedProject));
                final HashSet<String> personsSnapshotForNonCachedProject = new HashSet<>(cachedProjectMembers);
                allProjectCachedMembersByKey.put(nonCachedProject.getPublicKey(), personsSnapshotForNonCachedProject);
            }
        }
        return allProjectCachedMembersByKey.get(project.getPublicKey());
    }

    @NotNull
    @Override
    public ProjectFieldsContext forkSingle() {
        return forked.get();
    }

    @NotNull
    @Override
    public List<Campaign> getActiveCampaigns() {
        if (activeCampaigns == null) {
            throw new IllegalStateException("Active campaign was not loaded");
        }
        return activeCampaigns;
    }

}
