package ru.yandex.qe.dispenser.domain;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.builder.CompareToBuilder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiQuotaKey;
import ru.yandex.qe.dispenser.api.v1.DiQuotaType;
import ru.yandex.qe.dispenser.api.v1.DiQuotingMode;
import ru.yandex.qe.dispenser.api.v1.DiResourceType;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;

/**
 * Квота {@link Resource ресурса} на {@link Project проект} по заданной {@link QuotaSpec спецификации}
 * и разделенная на {@link Segment сегменты}, если для ресурса определена {@link ResourceSegmentation сегментация}.
 * <p>Квота определяется несколькими полями (значения которых указываются в базовых единицах измерения ресурса
 * ({@link DiResourceType#getBaseUnit()})):
 * <p>{@code max} - предел потребления для непосредственных подпроектов текущего проекта вместе с текущим проектом.
 * <p>{@code ownMax} - собственная квота на проект. Для листовых проектов существует соглашение {@code max == ownMax}.
 * <p>{@code ownActual} - собственное потребление квоты на проекте.
 * <p>{@code lastOverquotingTs} - время последнего превышения {@code max} суммой {@code ownActual} всех подпроектов вместе с
 * {@code ownActual} проекта. Обновляется в {@link ru.yandex.qe.dispenser.domain.lots.LotsManager}
 */
public final class Quota extends LongIndexBase implements Cloneable, Comparable<Quota> {
    public static final Quota FAKE = new Quota(
            new Builder(-1, new Key(
                    new QuotaSpec.Builder("fake",
                            new Resource(new Resource.Builder("fake", Service.withKey("fake").withName("fake").build())
                                    .name("fake")
                                    .mode(DiQuotingMode.ENTITIES_ONLY)
                                    .type(DiResourceType.MEMORY)
                                    .description("fake"))
                    ).description("fake")
                            .type(DiQuotaType.ABSOLUTE).build(),
                    new Project("fake", "fake", null),
                    Collections.emptySet()
            ))
    );

    @NotNull
    private final Key key;
    private final long max;
    private final long ownMax;
    private final long ownActual;

    // TODO: meta for quota
    @Nullable
    private final Long lastOverquotingTs;

    private Quota(@NotNull final Builder builder) {
        if (builder.id >= 0) {
            setId(builder.id);
        }
        this.key = builder.key;
        max = builder.max;
        ownMax = builder.ownMax;
        ownActual = builder.ownActual;

        lastOverquotingTs = builder.lastOverquotingTs;
    }

    @NotNull
    public static Builder builder(@NotNull final Quota quota) {
        return builder(quota, quota.key.project, quota.key.segments);
    }

    @NotNull
    public static Builder builder(@NotNull final Quota quota, @NotNull final Project project) {
        return builder(quota, project, quota.key.segments);
    }

    @NotNull
    public static Builder builder(@NotNull final Quota quota, @NotNull final Set<Segment> segments) {
        return builder(quota, quota.key.project, segments);
    }

    @NotNull
    public static Builder builder(@NotNull final Quota quota, @NotNull final Project project, @NotNull final Set<Segment> segments) {
        final Builder builder = builder(quota.getId(), quota.key.spec, project, segments);
        builder.max = quota.max;
        builder.ownMax = quota.ownMax;
        builder.ownActual = quota.ownActual;
        builder.lastOverquotingTs = quota.lastOverquotingTs;
        return builder;
    }

    @NotNull
    public static Builder builder(final long id, final @NotNull QuotaSpec spec, final @NotNull Project project,
                                  @NotNull final Set<Segment> segments) {
        return new Builder(id, new Key(spec, project, segments));
    }

    @NotNull
    public static Quota noQuota(@NotNull final Project project, final @NotNull QuotaSpec info,
                                @NotNull final Set<Segment> segments) {
        return builder(-1, info, project, segments).build();
    }

    @NotNull
    public static Quota noQuota(@NotNull final Quota.Key key) {
        return new Builder(-1, key).build();
    }

    @NotNull
    public static Quota noQuota(@NotNull final Project project, final @NotNull QuotaSpec info) {
        return new Builder(-1, Key.totalOf(info, project)).build();
    }

    public long getMax() {
        return max;
    }

    @NotNull
    public DiAmount getMaxAmount() {
        return DiAmount.of(getMax(), getResource().getType().getBaseUnit());
    }

    @NotNull
    public QuotaSpec getSpec() {
        return key.spec;
    }

    @NotNull
    public Resource getResource() {
        return getSpec().getResource();
    }

    @NotNull
    public Project getProject() {
        return key.project;
    }

    @Nullable
    public Long getLastOverquotingTs() {
        return lastOverquotingTs;
    }

    @NotNull
    public Set<Segment> getSegments() {
        return key.segments;
    }

    @NotNull
    public Key getKey() {
        return key;
    }

    public long getOwnMax() {
        return ownMax;
    }

    public long getOwnActual() {
        return ownActual;
    }

    @Override
    public int compareTo(@NotNull final Quota quota) {
        return new CompareToBuilder()
                .append(getSpec(), quota.getSpec())
                .append(getProject(), quota.getProject())
                .build();
    }

    @Override
    public boolean equals(@Nullable final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final Quota quota = (Quota) o;
        return key.equals(quota.key);
    }

    @Override
    public int hashCode() {
        return key.hashCode();
    }

    @Override
    public String toString() {
        return "Quota{" +
                "key=" + key +
                ", max=" + max +
                ", ownMax=" + ownMax +
                ", ownActual=" + ownActual +
                ", lastOverquotingTs=" + lastOverquotingTs +
                '}';
    }

    public static final class Builder {
        private final long id;
        private final Key key;

        private long max;
        private long ownMax;
        private long ownActual;

        @Nullable
        private Long lastOverquotingTs;

        private Builder(final long id, final @NotNull Key key) {
            this.id = id;
            this.key = key;
        }

        @NotNull
        public Builder max(final long max) {
//      updating of lots is external operation
//      updateLastOverquotingTsIf(actual <= this.max && max < actual);
            this.max = max;
            return this;
        }

        @NotNull
        public Builder lastOverquotingTs(@Nullable final Long lastOverquotingTs) {
            this.lastOverquotingTs = lastOverquotingTs;
            return this;
        }

        @NotNull
        public Builder ownMax(final long ownMax) {
            this.ownMax = ownMax;
            return this;
        }

        @NotNull
        public Builder ownActual(final long ownActual) {
            this.ownActual = ownActual;
            return this;
        }

        @NotNull
        public Builder change(final long value) {
//      TODO: quota below zero?
//      final long diff = value >= 0 ? value : -Math.min(this.actual, -value);
//      updating of lots is external operation
//      updateLastOverquotingTsIf(actual <= max && max < actual + diff);
            this.ownActual += value;

            return this;
        }

        @NotNull
        public Quota build() {
            return new Quota(this);
        }
    }

    public static class Key {
        @NotNull
        private final QuotaSpec spec;
        @NotNull
        private final Project project;
        @NotNull
        private final Set<Segment> segments;

        public Key(final @NotNull QuotaSpec spec, final @NotNull Project project,
                   final @NotNull Set<Segment> segments) {
            this.spec = spec;
            this.project = project;
            this.segments = segments;
        }

        public static Key totalOf(final @NotNull QuotaSpec spec, final @NotNull Project project) {
            final Set<Segmentation> segmentations = Hierarchy.get().getResourceSegmentationReader().getSegmentations(spec.getResource());
            final Set<Segment> segments = segmentations.stream()
                    .map(Segment::totalOf)
                    .collect(Collectors.toSet());

            return new Key(spec, project, segments);
        }

        @NotNull
        public QuotaSpec getSpec() {
            return spec;
        }

        @NotNull
        public Project getProject() {
            return project;
        }

        @NotNull
        public Set<Segment> getSegments() {
            return segments;
        }

        public Key withProject(final @NotNull Project project) {
            return new Key(spec, project, segments);
        }

        public Key withSegments(final @NotNull Set<Segment> segments) {
            return new Key(spec, project, segments);
        }

        public boolean isAggregation() {
            return getSegments().stream().anyMatch(Segment::isAggregationSegment);
        }

        public boolean isTotal() {
            return getSegments().stream().allMatch(Segment::isAggregationSegment);
        }

        public DiQuotaKey toView() {
            return new DiQuotaKey.Builder()
                    .projectKey(getProject().getPublicKey())
                    .serviceKey(getSpec().getResource().getService().getKey())
                    .resourceKey(getSpec().getResource().getKey().getPublicKey())
                    .quotaSpecKey(getSpec().getKey().getPublicKey())
                    .segmentKeys(SegmentUtils.getNonAggregationSegmentKeys(getSegments()))
                    .build();
        }

        @Override
        public boolean equals(@Nullable final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Key quota = (Key) o;
            return spec.equals(quota.spec) && project.equals(quota.project) && segments.equals(quota.segments);
        }

        @Override
        public int hashCode() {
            return Objects.hash(spec, project, segments);
        }

        public String humanReadable() {
            return "Quota key: (" +
                    "Project: \'" + getProject().getName() +
                    "\', Service: \'" + getSpec().getResource().getService().getName() +
                    "\', Resource: \'" + getSpec().getResource().getName() +
                    "\', Specification: \'" + getSpec().getDescription() +
                    "\', Segments: [" + segments.stream().map(Segment::getName).filter(Objects::nonNull).collect(Collectors.joining(",")) +
                    "])";
        }

        @Override
        public String toString() {
            return "Quota.Key{" +
                    "project=" + getProject().getKey() +
                    ", service=" + getSpec().getResource().getService().getKey() +
                    ", resource=" + getSpec().getResource().getKey().getPublicKey() +
                    ", specification=" + getSpec().getKey().getPublicKey() +
                    ", segments=" + getSegments() +
                    '}';
        }
    }

    public static class ChangeHolder {
        private final Map<Key, Long> maxByKey = new HashMap<>();
        private final Map<Key, Long> ownActualByKey = new HashMap<>();
        private final Map<Key, Long> ownMaxByKey = new HashMap<>();

        public Set<Key> getQuotaKeysWithMaxChanges() {
            return Sets.union(maxByKey.keySet(), ownMaxByKey.keySet());
        }

        public Set<Key> getChangedQuotaKeys() {
            return ImmutableSet.<Key>builder()
                    .addAll(maxByKey.keySet())
                    .addAll(ownMaxByKey.keySet())
                    .addAll(ownActualByKey.keySet())
                    .build();
        }

        public void setMax(final Key quotaKey, final Long max) {
            maxByKey.put(quotaKey, max);
        }

        public void setOwnMax(final Key quotaKey, final Long ownMax) {
            ownMaxByKey.put(quotaKey, ownMax);
        }

        public void setMaxes(final Map<Key, Long> maxes) {
            maxByKey.putAll(maxes);
        }

        public void setOwnMaxes(final Map<Key, Long> ownMaxes) {
            ownMaxByKey.putAll(ownMaxes);
        }

        public void setActual(final Key quotaKey, final Long actual) {
            ownActualByKey.put(quotaKey, actual);
        }

        public Map<Key, Long> getMax() {
            return maxByKey;
        }

        public Long getMax(final Key quotaKey) {
            return maxByKey.get(quotaKey);
        }

        public Map<Key, Long> getOwnActual() {
            return ownActualByKey;
        }

        public Map<Key, Long> getOwnMax() {
            return ownMaxByKey;
        }

        public Long getOwnMax(final Key quotaKey) {
            return ownMaxByKey.get(quotaKey);
        }

        public Set<Quota> applyToQuotas(final Set<Quota> quotasBySpecs) {

            final Set<Quota.Key> modifiedQuotaKeys = getChangedQuotaKeys();

            return quotasBySpecs.stream()
                    .map(quota -> {
                        final Key key = quota.getKey();
                        if (modifiedQuotaKeys.contains(key)) {
                            final Builder builder = builder(quota);

                            builder.max(maxByKey.getOrDefault(key, quota.getMax()));
                            builder.ownMax(ownMaxByKey.getOrDefault(key, quota.getOwnMax()));
                            builder.ownActual(ownActualByKey.getOrDefault(key, quota.getOwnActual()));

                            return builder.build();
                        }
                        return quota;
                    })
                    .collect(Collectors.toSet());
        }
    }
}
