package ru.yandex.qe.dispenser.domain;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequest;
import ru.yandex.qe.dispenser.api.v1.DiResourcePreorderReasonType;
import ru.yandex.qe.dispenser.domain.dao.goal.Goal;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.index.LongIndexable;
import ru.yandex.qe.dispenser.domain.util.BigOrderLegacy;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;

/**
 * Заявка на изменение квоты. Подразделяется на несколько {@link Type типов}, в зависимости от которых проходит по определенному
 * воркфлоу.
 */
@ParametersAreNonnullByDefault
public class QuotaChangeRequest extends LongIndexBase {
    @NotNull
    private final Person author;
    @NotNull
    private final Project project;
    @Nullable
    private final String description;
    @Nullable
    private final String comment;
    @Nullable
    private final String calculations;
    private final long created;
    private final long updated;
    @NotNull
    private final Status status;
    @Nullable
    private final String trackerIssueKey;
    @NotNull
    private final List<Change> changes;
    @NotNull
    private final Type type;
    @Nullable
    private final Project sourceProject;
    @NotNull
    private final List<String> chartLinks;
    @Nullable
    private final String chartLinksAbsenceExplanation;
    @Nullable
    private final Map<String, String> additionalProperties;
    @Nullable
    private final Campaign campaign;
    @Nullable
    private final DiResourcePreorderReasonType resourcePreorderReasonType;
    @Nullable
    private final Goal goal;
    private final boolean readyForAllocation;
    @Nullable
    private final Map<Long, String> requestGoalAnswers;
    @Nullable
    private final String summary;
    private final boolean showAllocationNote;
    private final boolean unbalanced;

    private final double cost;
    private final long requestOwningCost;
    private final boolean importantRequest;
    @Nullable
    private final ru.yandex.qe.dispenser.domain.Campaign.Type campaignType;

    public QuotaChangeRequest(final Builder builder) {
        if (builder.id != null) {
            setId(builder.id);
        }
        this.project = ValidationUtils.requireNonNull(builder.project, "Project is required");
        this.changes = ValidationUtils.requireNonNull(builder.changes, "Changes are required");
        this.author = ValidationUtils.requireNonNull(builder.author, "Author is required");
        this.description = builder.description;
        this.comment = builder.comment;
        this.calculations = builder.calculations;
        this.created = ValidationUtils.requireNonNull(builder.created, "Created is required");
        this.updated = ValidationUtils.requireNonNull(builder.updated, "Updated is required");
        this.status = ValidationUtils.requireNonNull(builder.status, "Status is required");
        this.trackerIssueKey = builder.trackerIssueKey;
        this.type = ValidationUtils.requireNonNull(builder.type, "Type is required");
        this.sourceProject = builder.sourceProject;
        this.chartLinks = ValidationUtils.requireNonNull(builder.chartLinks, "Chart links are required");
        this.chartLinksAbsenceExplanation = builder.chartLinksAbsenceExplanation;
        this.additionalProperties = Collections.unmodifiableMap(builder.additionalProperties);
        this.campaign = builder.campaign;
        this.resourcePreorderReasonType = builder.resourcePreorderReasonType;
        this.goal = builder.goal;
        this.readyForAllocation = builder.readyForAllocation;
        this.requestGoalAnswers = builder.requestGoalAnswers;
        this.cost = ValidationUtils.requireNonNull(builder.cost, "Cost is required");
        this.summary = builder.summary;
        this.showAllocationNote = builder.showAllocationNote;
        this.requestOwningCost = builder.requestOwningCost;
        this.importantRequest = builder.importantRequest;
        this.unbalanced = builder.unbalanced;
        this.campaignType = builder.campaignType;
    }

    @NotNull
    public Project getProject() {
        return project;
    }

    @NotNull
    public Person getAuthor() {
        return author;
    }

    @NotNull
    public List<Change> getChanges() {
        return changes;
    }

    @Nullable
    public String getDescription() {
        return description;
    }

    @Nullable
    public String getComment() {
        return comment;
    }

    @Nullable
    public String getCalculations() {
        return calculations;
    }

    public long getCreated() {
        return created;
    }

    public long getUpdated() {
        return updated;
    }

    @NotNull
    public Status getStatus() {
        return status;
    }

    @NotNull
    public Type getType() {
        return type;
    }

    @Nullable
    public Project getSourceProject() {
        return sourceProject;
    }

    @Nullable
    public String getTrackerIssueKey() {
        return trackerIssueKey;
    }

    @NotNull
    public List<String> getChartLinks() {
        return chartLinks;
    }

    @Nullable
    public String getChartLinksAbsenceExplanation() {
        return chartLinksAbsenceExplanation;
    }

    @Nullable
    public Map<String, String> getAdditionalProperties() {
        return additionalProperties;
    }

    @Nullable
    public Campaign getCampaign() {
        return campaign;
    }

    @Nullable
    public Long getCampaignId() {
        return campaign != null ? campaign.getId() : null;
    }

    @Nullable
    public DiResourcePreorderReasonType getResourcePreorderReasonType() {
        return resourcePreorderReasonType;
    }

    @Nullable
    public Goal getGoal() {
        return goal;
    }

    public boolean isReadyForAllocation() {
        return readyForAllocation;
    }

    @Nullable
    public Map<Long, String> getRequestGoalAnswers() {
        return requestGoalAnswers;
    }

    public double getCost() {
        return cost;
    }

    @Nullable
    public String getSummary() {
        return summary;
    }

    public boolean showAllocationNote() {
        return showAllocationNote;
    }

    public long getRequestOwningCost() {
        return requestOwningCost;
    }

    public boolean isImportantRequest() {
        return importantRequest;
    }

    public boolean isUnbalanced() {
        return unbalanced;
    }

    @Nullable
    public ru.yandex.qe.dispenser.domain.Campaign.Type getCampaignType() {
        return campaignType;
    }

    public Builder copyBuilder() {
        final Builder builder = new Builder()
                .project(project)
                .changes(changes)
                .author(author)
                .description(description)
                .comment(comment)
                .calculations(calculations)
                .created(created)
                .updated(updated)
                .status(status)
                .type(type)
                .trackerIssueKey(trackerIssueKey)
                .sourceProject(sourceProject)
                .chartLinks(chartLinks)
                .chartLinksAbsenceExplanation(chartLinksAbsenceExplanation)
                .additionalProperties(additionalProperties)
                .campaign(campaign)
                .resourcePreorderReasonType(resourcePreorderReasonType)
                .goal(goal)
                .readyForAllocation(readyForAllocation)
                .requestGoalAnswers(requestGoalAnswers)
                .cost(cost)
                .summary(summary)
                .showAllocationNote(showAllocationNote)
                .requestOwningCost(requestOwningCost)
                .importantRequest(importantRequest)
                .unbalanced(unbalanced)
                .campaignType(campaignType);

        if (getId() >= 0) {
            builder.id(getId());
        }

        return builder;
    }

    public DiQuotaChangeRequest toBasicView() {
        return new DiQuotaChangeRequest.Builder(getId())
                .project(project.toView(false))
                .changes(changes.stream().map(Change::toView).collect(Collectors.toList()))
                .author(author.getLogin())
                .description(description)
                .comment(comment)
                .calculations(calculations)
                .created(created)
                .updated(updated)
                .status(status.toView())
                .type(type.toView())
                .trackerIssueKey(trackerIssueKey)
                .sourceProject(sourceProject != null ? sourceProject.toView(false) : null)
                .chartLinks(chartLinks)
                .chartLinksAbsenceExplanation(chartLinksAbsenceExplanation)
                .additionalProperties(additionalProperties)
                .campaign(campaign != null ? campaign.toView() : null)
                .resourcePreorderReasonType(resourcePreorderReasonType)
                .goal(goal != null ? goal.toView() : null)
                .readyForAllocation(readyForAllocation)
                .requestGoalAnswers(requestGoalAnswers)
                .cost(cost)
                .summary(summary)
                .showAllocationNote(showAllocationNote)
                .requestOwningCost(String.valueOf(requestOwningCost))
                .importantRequest(importantRequest)
                .unbalanced(unbalanced)
                .build();
    }

    @Override
    public String toString() {
        return "QuotaChangeRequest{" +
                "author=" + author +
                ", project=" + project +
                ", description='" + description + '\'' +
                ", comment='" + comment + '\'' +
                ", calculations='" + calculations + '\'' +
                ", created=" + created +
                ", updated=" + updated +
                ", status=" + status +
                ", trackerIssueKey='" + trackerIssueKey + '\'' +
                ", changes=" + changes +
                ", type=" + type +
                ", sourceProject=" + sourceProject +
                ", chartLinks=" + chartLinks +
                ", chartLinksAbsenceExplanation='" + chartLinksAbsenceExplanation + '\'' +
                ", additionalProperties=" + additionalProperties +
                ", campaign=" + campaign +
                ", resourcePreorderReasonType=" + resourcePreorderReasonType +
                ", goal=" + goal +
                ", readyForAllocation=" + readyForAllocation +
                ", requestGoalAnswers=" + requestGoalAnswers +
                ", summary='" + summary + '\'' +
                ", showAllocationNote=" + showAllocationNote +
                ", unbalanced=" + unbalanced +
                ", cost=" + cost +
                ", requestOwningCost=" + requestOwningCost +
                ", importantRequest=" + importantRequest +
                "} " + super.toString();
    }

    public enum Status {
        NEW(DiQuotaChangeRequest.Status.NEW, DiQuotaChangeRequest.Permission.CAN_REOPEN),
        CANCELLED(DiQuotaChangeRequest.Status.CANCELLED, DiQuotaChangeRequest.Permission.CAN_CANCEL),
        REJECTED(DiQuotaChangeRequest.Status.REJECTED, DiQuotaChangeRequest.Permission.CAN_REJECT),
        CONFIRMED(DiQuotaChangeRequest.Status.CONFIRMED, DiQuotaChangeRequest.Permission.CAN_CONFIRM),

        READY_FOR_REVIEW(DiQuotaChangeRequest.Status.READY_FOR_REVIEW, DiQuotaChangeRequest.Permission.CAN_MARK_AS_READY_FOR_REVIEW),
        COMPLETED(DiQuotaChangeRequest.Status.COMPLETED, DiQuotaChangeRequest.Permission.CAN_COMPLETE),

        APPLIED(DiQuotaChangeRequest.Status.APPLIED, DiQuotaChangeRequest.Permission.CAN_APPLY),
        NEED_INFO(DiQuotaChangeRequest.Status.NEED_INFO, DiQuotaChangeRequest.Permission.CAN_MARK_AS_NEED_INFO),
        APPROVED(DiQuotaChangeRequest.Status.APPROVED, DiQuotaChangeRequest.Permission.CAN_APPROVE),
        ;

        @NotNull
        private final DiQuotaChangeRequest.Status view;
        private final DiQuotaChangeRequest.Permission permissionView;

        Status(final DiQuotaChangeRequest.Status view, final DiQuotaChangeRequest.Permission permissionView) {
            this.view = view;
            this.permissionView = permissionView;
        }

        @NotNull
        public DiQuotaChangeRequest.Status toView() {
            return view;
        }

        @NotNull
        public static Status of(final DiQuotaChangeRequest.Status statusView) {
            for (final Status value : values()) {
                if (value.view == statusView) {
                    return value;
                }
            }
            throw new IllegalArgumentException("Invalid status");
        }

        @NotNull
        public DiQuotaChangeRequest.Permission getPermissionView() {
            return permissionView;
        }
    }

    public enum Type {
        QUOTA_INCREASE(DiQuotaChangeRequest.Type.QUOTA_INCREASE),
        RESOURCE_PREORDER(DiQuotaChangeRequest.Type.RESOURCE_PREORDER),
        QUOTA_MOVE(DiQuotaChangeRequest.Type.QUOTA_MOVE);

        @NotNull
        private final DiQuotaChangeRequest.Type diType;

        Type(final DiQuotaChangeRequest.Type diType) {
            this.diType = diType;
        }

        @NotNull
        public DiQuotaChangeRequest.Type toView() {
            return diType;
        }

        @NotNull
        public static Type of(final DiQuotaChangeRequest.Type typeView) {
            for (final Type type : values()) {
                if (type.diType == typeView) {
                    return type;
                }
            }
            throw new IllegalArgumentException("Invalid type");
        }
    }

    public enum Field {
        STATUS("status", req -> req.getStatus().toView()),
        CHANGES("changes", req -> req.getChanges().stream().map(Change::toView).collect(Collectors.toList())),
        DESCRIPTION("description", QuotaChangeRequest::getDescription),
        COMMENT("comment", QuotaChangeRequest::getComment),
        CALCULATION("calculations", QuotaChangeRequest::getCalculations),
        CHART_LINKS("chartLinks", QuotaChangeRequest::getChartLinks),
        CHART_LINKS_ABSENCE_EXPLANATION("chartLinksAbsenceExplanation", QuotaChangeRequest::getChartLinksAbsenceExplanation),
        ADDITIONAL_PROPERTIES("additionalProperties", QuotaChangeRequest::getAdditionalProperties),
        GOAL_ID("goal", req -> Optional.ofNullable(req.getGoal()).map(Goal::toView).orElse(null)),
        RESOURCE_PREORDER_REASON_TYPE("resourcePreorderReasonType",
                req -> Optional.ofNullable(req.getResourcePreorderReasonType()).orElse(null)),
        PROJECT("project", req -> req.getProject().toView(false)),
        ISSUE_KEY("trackerIssueKey", QuotaChangeRequest::getTrackerIssueKey),
        READY_FOR_ALLOCATION("readyForAllocation", QuotaChangeRequest::isReadyForAllocation),
        REQUEST_GOAL_ANSWERS("requestGoalAnswers", QuotaChangeRequest::getRequestGoalAnswers),
        SUMMARY("summary", QuotaChangeRequest::getSummary),
        IMPORTANT_REQUEST("importantRequest", QuotaChangeRequest::isImportantRequest),
        UNBALANCED("unbalanced", QuotaChangeRequest::isUnbalanced),
        ;

        private final String fieldName;
        private final Function<QuotaChangeRequest, Object> viewProvider;

        Field(final String fieldName, final Function<QuotaChangeRequest, Object> viewProvider) {
            this.fieldName = fieldName;
            this.viewProvider = viewProvider;
        }

        public String getFieldName() {
            return fieldName;
        }

        public Object getView(final QuotaChangeRequest request) {
            return viewProvider.apply(request);
        }
    }

    public static class Builder {
        @Nullable
        private Long id;
        @Nullable
        private Project project;
        @Nullable
        private List<Change> changes;
        @Nullable
        private Person author;
        @Nullable
        private String description;
        @Nullable
        private String comment;
        @Nullable
        private String calculations;
        @Nullable
        private Long created;
        @Nullable
        private Long updated;
        @Nullable
        private Status status;
        @Nullable
        private String trackerIssueKey;
        @Nullable
        private Type type;
        @Nullable
        private Project sourceProject;
        @Nullable
        private List<String> chartLinks;
        @Nullable
        private String chartLinksAbsenceExplanation;
        @NotNull
        public Map<String, String> additionalProperties = new HashMap<>();
        @Nullable
        private Campaign campaign;
        @Nullable
        private DiResourcePreorderReasonType resourcePreorderReasonType;
        @Nullable
        private Goal goal;
        private boolean readyForAllocation;
        @Nullable
        private Map<Long, String> requestGoalAnswers;
        @Nullable
        private Double cost;
        @Nullable
        private String summary;
        private boolean showAllocationNote;
        private Long requestOwningCost;
        private boolean importantRequest;
        private boolean unbalanced;
        @Nullable
        private ru.yandex.qe.dispenser.domain.Campaign.Type campaignType;

        @NotNull
        public Builder id(final long id) {
            this.id = id;
            return this;
        }

        @NotNull
        public Builder project(final Project project) {
            this.project = project;
            return this;
        }

        @NotNull
        public Builder changes(final List<Change> changes) {
            this.changes = changes;
            return this;
        }

        @NotNull
        public Builder author(final Person author) {
            this.author = author;
            return this;
        }

        @NotNull
        public Builder description(@Nullable final String description) {
            this.description = description;
            return this;
        }

        @NotNull
        public Builder comment(@Nullable final String comment) {
            this.comment = comment;
            return this;
        }

        @NotNull
        public Builder calculations(final String calculations) {
            this.calculations = calculations;
            return this;
        }

        @NotNull
        public Builder created(final long created) {
            this.created = created;
            return this;
        }

        @NotNull
        public Builder updated(final long updated) {
            this.updated = updated;
            return this;
        }

        @NotNull
        public Builder status(final Status status) {
            this.status = status;
            return this;
        }

        @NotNull
        public Builder trackerIssueKey(@Nullable final String trackerIssueKey) {
            this.trackerIssueKey = trackerIssueKey;
            return this;
        }

        @NotNull
        public Builder type(final Type type) {
            this.type = type;
            return this;
        }

        @NotNull
        public Builder sourceProject(@Nullable final Project project) {
            this.sourceProject = project;
            return this;
        }

        @NotNull
        public Builder chartLinks(final List<String> chartLinks) {
            this.chartLinks = chartLinks;
            return this;
        }

        @NotNull
        public Builder chartLinksAbsenceExplanation(@Nullable final String chartLinksAbsenceExplanation) {
            this.chartLinksAbsenceExplanation = chartLinksAbsenceExplanation;
            return this;
        }

        @NotNull
        public Builder resourcePreorderReasonType(@Nullable final DiResourcePreorderReasonType resourcePreorderReasonType) {
            this.resourcePreorderReasonType = resourcePreorderReasonType;
            return this;
        }

        @NotNull
        public Builder goal(@Nullable final Goal goal) {
            this.goal = goal;
            return this;
        }

        @NotNull
        public Builder additionalProperties(@Nullable final Map<String, String> properties) {
            if (properties != null) {
                this.additionalProperties = properties;
            }
            return this;
        }

        @NotNull
        public Builder campaign(@Nullable final Campaign campaign) {
            this.campaign = campaign;
            return this;
        }


        @NotNull
        public Builder readyForAllocation(final boolean readyForAllocation) {
            this.readyForAllocation = readyForAllocation;
            return this;
        }

        @NotNull
        public Builder requestGoalAnswers(@Nullable final Map<Long, String> requestGoalAnswers) {
            this.requestGoalAnswers = requestGoalAnswers;
            return this;
        }

        @NotNull
        public Builder cost(final double cost) {
            this.cost = cost;
            return this;
        }

        @NotNull
        public Builder summary(@Nullable final String summary) {
            this.summary = summary;
            return this;
        }

        @NotNull
        public Builder showAllocationNote(final boolean showAllocationNote) {
            this.showAllocationNote = showAllocationNote;
            return this;
        }

        @NotNull
        public Builder requestOwningCost(final Long requestOwningCost) {
            this.requestOwningCost = requestOwningCost;
            return this;
        }

        public Builder importantRequest(final boolean importantRequest) {
            this.importantRequest = importantRequest;
            return this;
        }

        public Builder unbalanced(final boolean unbalanced) {
            this.unbalanced = unbalanced;
            return this;
        }

        public Builder campaignType(@Nullable ru.yandex.qe.dispenser.domain.Campaign.Type campaignType) {
            this.campaignType = campaignType;
            return this;
        }

        @NotNull
        public QuotaChangeRequest build() {
            return new QuotaChangeRequest(this);
        }
    }

    public static class ChangeKey {
        @Nullable
        private final BigOrder bigOrder;
        @NotNull
        private final Resource resource;
        @NotNull
        private final Set<Segment> segments;

        @NotNull
        public static ChangeKey fromBodyValues(final String serviceKey,
                                               final String resourceKey,
                                               final Set<String> segmentKeys,
                                               @Nullable final BigOrder bigOrder) {
            final Service service = Hierarchy.get().getServiceReader().read(serviceKey);
            final Resource resource = Hierarchy.get().getResourceReader().read(new Resource.Key(resourceKey, service));
            final Set<Segment> segments = SegmentUtils.getCompleteSegmentSet(resource, segmentKeys);

            return new ChangeKey(bigOrder, resource, segments);
        }


        public ChangeKey(@Nullable final BigOrder bigOrder,
                         @NotNull final Resource resource,
                         @NotNull final Set<Segment> segments) {
            this.bigOrder = bigOrder;
            this.resource = resource;
            this.segments = segments;
        }

        @NotNull
        public Resource getResource() {
            return resource;
        }

        @NotNull
        public Set<Segment> getSegments() {
            return segments;
        }

        @Nullable
        public BigOrder getBigOrder() {
            return bigOrder;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final ChangeKey changeKey = (ChangeKey) o;
            return Objects.equals(bigOrder, changeKey.bigOrder) &&
                    resource.equals(changeKey.resource) &&
                    segments.equals(changeKey.segments);
        }

        @Override
        public int hashCode() {
            return Objects.hash(bigOrder, resource, segments);
        }

        @Override
        public String toString() {
            return "ChangeKey{" +
                    "bigOrder=" + bigOrder +
                    ", resource=" + resource +
                    ", segments=" + segments +
                    '}';
        }

    }

    public static class ChangeAmount {
        @NotNull
        protected final ChangeKey changeKey;
        protected final long amount;

        public ChangeAmount(@NotNull final ChangeKey changeKey, final long amount) {
            this.changeKey = changeKey;
            this.amount = amount;
        }

        @NotNull
        public ChangeKey getKey() {
            return changeKey;
        }

        public long getAmount() {
            return amount;
        }

        @NotNull
        public Resource getResource() {
            return changeKey.getResource();
        }

        @NotNull
        public Set<Segment> getSegments() {
            return changeKey.getSegments();
        }

        @Nullable
        public BigOrder getBigOrder() {
            return changeKey.getBigOrder();
        }
    }

    public static class Change extends ChangeAmount implements LongIndexable {
        private long id = -1;
        private final long amountReady;
        private final long amountAllocated;
        private final long amountAllocating;
        @NotNull
        private final BigDecimal owningCost;

        Change(final ChangeKey changeKey, final long amount, final long amountReady, final long amountAllocated,
               final long amountAllocating, final BigDecimal owningCost) {
            super(changeKey, amount);
            this.amountReady = amountReady;
            this.amountAllocated = amountAllocated;
            this.amountAllocating = amountAllocating;
            this.owningCost = owningCost;
        }

        public void setId(final long id) {
            this.id = id;
        }

        @Override
        public long getId() {
            return id;
        }

        public long getAmountReady() {
            return amountReady;
        }

        public long getAmountAllocated() {
            return amountAllocated;
        }

        public long getAmountAllocating() {
            return amountAllocating;
        }

        @NotNull
        public BigDecimal getOwningCost() {
            return owningCost;
        }

        public boolean hasReadyOrAllocatedOrAllocating() {
            return amountReady > 0 || amountAllocated > 0 || amountAllocating > 0;
        }

        public static QuotaRequestChangeBuilder builder() {
            return new QuotaRequestChangeBuilder();
        }

        public static QuotaRequestChangeBuilder newChangeBuilder() {
            return new QuotaRequestChangeBuilder()
                    .amountAllocated(0)
                    .amountReady(0)
                    .amountAllocating(0)
                    .owningCost(BigDecimal.ZERO);
        }

        public QuotaRequestChangeBuilder copyBuilder() {
            return new QuotaRequestChangeBuilder()
                    .id(getId())
                    .key(changeKey)
                    .amount(amount)
                    .amountReady(amountReady)
                    .amountAllocated(amountAllocated)
                    .amountAllocating(amountAllocating)
                    .owningCost(owningCost);
        }

        public DiQuotaChangeRequest.Change toView() {
            final Resource resource = changeKey.getResource();
            return new DiQuotaChangeRequest.Change(
                    changeKey.getBigOrder() == null ? null : changeKey.getBigOrder().toView(),
                    new DiQuotaChangeRequest.Service(resource.getService().getKey(), resource.getService().getName()),
                    new DiQuotaChangeRequest.Resource(resource.getPublicKey(), resource.getName()),
                    changeKey.getSegments().stream().map(Segment::getPublicKey).collect(Collectors.toSet()),
                    DiAmount.of(this.amount, resource.getType().getBaseUnit()),
                    DiAmount.of(this.amountReady, resource.getType().getBaseUnit()),
                    DiAmount.of(this.amountAllocated, resource.getType().getBaseUnit()),
                    DiAmount.of(this.amountAllocating, resource.getType().getBaseUnit()),
                    owningCost.toString(), // this method used only for history / event fields, change to output format, if needed
                    null // this method used only for history / event fields, change to output format, if needed
            );
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Change change = (Change) o;
            return amount == change.amount &&
                    changeKey.equals(change.changeKey);
        }

        @Override
        public int hashCode() {
            return Objects.hash(changeKey, amount);
        }

        @Override
        public String toString() {
            return changeKey.getResource().getService() + "_" +
                    changeKey.getResource().getPublicKey() + ":" +
                    (changeKey.getBigOrder() == null ? "" : changeKey.getBigOrder().getDate() + ":") +
                    changeKey.getSegments() + ":{" +
                    "amount=" + amount +
                    ", amountReady=" + amountReady +
                    ", amountAllocated=" + amountAllocated +
                    ", amountAllocating=" + amountAllocating +
                    ", owningCost=" + owningCost +
                    ", id=" + id +
                    '}';
        }
    }

    public static class BigOrder {

        private final long id;
        @NotNull
        private final LocalDate date;
        private final boolean inCampaign;

        public BigOrder(final long id, @NotNull final LocalDate date, final boolean inCampaign) {
            this.id = id;
            this.date = date;
            this.inCampaign = inCampaign;
        }

        public long getId() {
            return id;
        }

        @NotNull
        public LocalDate getDate() {
            return date;
        }

        public boolean isInCampaign() {
            return inCampaign;
        }

        @NotNull
        public DiQuotaChangeRequest.Order toView() {
            // TODO Get rid of isValid
            return new DiQuotaChangeRequest.Order(id, date, BigOrderLegacy.checkValid(date));
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final BigOrder bigOrder = (BigOrder) o;
            return id == bigOrder.id;
        }

        @Override
        public int hashCode() {
            return Objects.hash(id);
        }

        @Override
        public String toString() {
            return "BigOrder{"
                    + "id=" + id
                    + ", date='" + date + '\''
                    + ", inCampaign=" + inCampaign
                    + '}';
        }

    }

    public static class Campaign {

        private final long id;
        @NotNull
        private final String key;
        @NotNull
        private final String name;
        @NotNull
        private final ru.yandex.qe.dispenser.domain.Campaign.Status status;
        @Deprecated
        private final boolean requestModificationDisabledForNonManagers;
        private final boolean deleted;
        private final boolean allowedRequestModificationWhenClosed;
        private final boolean allowedModificationOnMissingAdditionalFields;
        @Deprecated
        private final boolean forcedAllowingRequestStatusChange;
        private final boolean singleProviderRequestModeEnabled;
        @Deprecated
        private final boolean allowedRequestCreationForProviderAdmin;
        private final boolean requestCreationDisabled;
        @Deprecated
        private final boolean allowedRequestCreationForCapacityPlanner;
        @NotNull
        private final ru.yandex.qe.dispenser.domain.Campaign.Type type;
        private final boolean requestModificationDisabled;

        public static Campaign from(final ru.yandex.qe.dispenser.domain.Campaign campaign) {
            return new Campaign(campaign.getId(),
                    campaign.getKey(),
                    campaign.getName(),
                    campaign.getStatus(),
                    campaign.isRequestModificationDisabledForNonManagers(),
                    false,
                    campaign.isAllowedRequestModificationWhenClosed(),
                    campaign.isAllowedModificationOnMissingAdditionalFields(),
                    campaign.isForcedAllowingRequestStatusChange(),
                    campaign.isSingleProviderRequestModeEnabled(),
                    campaign.isAllowedRequestCreationForProviderAdmin(),
                    campaign.isRequestCreationDisabled(),
                    campaign.isAllowedRequestCreationForCapacityPlanner(),
                    campaign.getType(),
                    campaign.isRequestModificationDisabled());
        }

        @SuppressWarnings({"BooleanParameter", "ConstructorWithTooManyParameters"})
        public Campaign(final long id, @NotNull final String key, @NotNull final String name,
                        @NotNull final ru.yandex.qe.dispenser.domain.Campaign.Status status,
                        @Deprecated
                        final boolean requestModificationDisabledForNonManagers, final boolean deleted,
                        final boolean allowedRequestModificationWhenClosed,
                        final boolean allowedModificationOnMissingAdditionalFields,
                        @Deprecated
                        final boolean forcedAllowingRequestStatusChange,
                        final boolean singleProviderRequestModeEnabled,
                        @Deprecated
                        final boolean allowedRequestCreationForProviderAdmin,
                        final boolean requestCreationDisabled,
                        @Deprecated
                        final boolean allowedRequestCreationForCapacityPlanner,
                        @NotNull final ru.yandex.qe.dispenser.domain.Campaign.Type type,
                        boolean requestModificationDisabled) {
            this.id = id;
            this.key = key;
            this.name = name;
            this.status = status;
            this.requestModificationDisabledForNonManagers = requestModificationDisabledForNonManagers;
            this.deleted = deleted;
            this.allowedRequestModificationWhenClosed = allowedRequestModificationWhenClosed;
            this.allowedModificationOnMissingAdditionalFields = allowedModificationOnMissingAdditionalFields;
            this.forcedAllowingRequestStatusChange = forcedAllowingRequestStatusChange;
            this.singleProviderRequestModeEnabled = singleProviderRequestModeEnabled;
            this.allowedRequestCreationForProviderAdmin = allowedRequestCreationForProviderAdmin;
            this.requestCreationDisabled = requestCreationDisabled;
            this.allowedRequestCreationForCapacityPlanner = allowedRequestCreationForCapacityPlanner;
            this.type = type;
            this.requestModificationDisabled = requestModificationDisabled;
        }

        public long getId() {
            return id;
        }

        @NotNull
        public String getKey() {
            return key;
        }

        @NotNull
        public String getName() {
            return name;
        }

        @NotNull
        public ru.yandex.qe.dispenser.domain.Campaign.Status getStatus() {
            return status;
        }

        @Deprecated
        public boolean isRequestModificationDisabledForNonManagers() {
            return requestModificationDisabledForNonManagers;
        }

        public boolean isDeleted() {
            return deleted;
        }

        public boolean isAllowedRequestModificationWhenClosed() {
            return allowedRequestModificationWhenClosed;
        }

        public boolean isAllowedModificationOnMissingAdditionalFields() {
            return allowedModificationOnMissingAdditionalFields;
        }

        @Deprecated
        public boolean isForcedAllowingRequestStatusChange() {
            return forcedAllowingRequestStatusChange;
        }

        public boolean isSingleProviderRequestModeEnabled() {
            return singleProviderRequestModeEnabled;
        }

        @Deprecated
        public boolean isAllowedRequestCreationForProviderAdmin() {
            return allowedRequestCreationForProviderAdmin;
        }

        public boolean isRequestCreationDisabled() {
            return requestCreationDisabled;
        }

        @Deprecated
        public boolean isAllowedRequestCreationForCapacityPlanner() {
            return allowedRequestCreationForCapacityPlanner;
        }

        @NotNull
        public ru.yandex.qe.dispenser.domain.Campaign.Type getType() {
            return type;
        }

        public boolean isRequestModificationDisabled() {
            return requestModificationDisabled;
        }

        public DiQuotaChangeRequest.Campaign toView() {
            return new DiQuotaChangeRequest.Campaign(id, key, name, status.name(), type.name());
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Campaign campaign = (Campaign) o;
            return id == campaign.id;
        }

        @Override
        public int hashCode() {
            return Objects.hash(id);
        }

        @Override
        public String toString() {
            return "Campaign{" +
                    "id=" + id +
                    ", key='" + key + '\'' +
                    ", name='" + name + '\'' +
                    ", status=" + status +
                    ", requestModificationDisabledForNonManagers=" + requestModificationDisabledForNonManagers +
                    ", deleted=" + deleted +
                    ", allowedRequestModificationWhenClosed=" + allowedRequestModificationWhenClosed +
                    ", allowedModificationOnMissingAdditionalFields=" + allowedModificationOnMissingAdditionalFields +
                    ", forcedAllowingRequestStatusChange=" + forcedAllowingRequestStatusChange +
                    ", singleProviderRequestModeEnabled=" + singleProviderRequestModeEnabled +
                    ", allowedRequestCreationForProviderAdmin=" + allowedRequestCreationForProviderAdmin +
                    ", allowedRequestCreationForCapacityPlanner=" + allowedRequestCreationForCapacityPlanner +
                    ", type=" + type +
                    ", requestModificationDisabled=" + requestModificationDisabled +
                    '}';
        }

    }

}
