package ru.yandex.qe.dispenser.domain;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiCheckValue;
import ru.yandex.qe.dispenser.api.v1.DiCheckValueType;
import ru.yandex.qe.dispenser.api.v1.DiUnit;

@ParametersAreNonnullByDefault
public abstract class QuotaCompareValue {

    @NotNull
    public static QuotaCompareValue of(final DiCheckValue<?> value) {
        if (value.getType() == DiCheckValueType.ABSOLUTE) {
            return new QuotaAbsoluteValue((DiAmount) value.getValue());
        }
        return new QuotaPercentValue((Number) value.getValue());
    }

    public abstract boolean test(QuotaView quota, QuotaCompareType compareType);

    private static class QuotaAbsoluteValue extends QuotaCompareValue {

        final DiAmount amount;

        private QuotaAbsoluteValue(final DiAmount amount) {
            this.amount = amount;
        }

        @Override
        public boolean test(final QuotaView quota, final QuotaCompareType compareType) {
            final DiUnit baseUnit = quota.getResource().getType().getBaseUnit();
            final long convertedAmount = baseUnit.convert(amount);

            return compareType.test(quota.getTotalActual(), convertedAmount);
        }

        @Override
        public String toString() {
            return amount.humanize().toString();
        }
    }

    private static class QuotaPercentValue extends QuotaCompareValue {
        final Number amount;

        private QuotaPercentValue(final Number amount) {
            this.amount = amount;
        }

        @Override
        public boolean test(final QuotaView quota, final QuotaCompareType compareType) {
            final double actualUsage = 100 * (double) quota.getTotalActual() / quota.getMax();
            return compareType.test(actualUsage, amount);
        }

        @Override
        public String toString() {
            return amount + "%";
        }
    }
}
