package ru.yandex.qe.dispenser.domain;

import java.util.Optional;
import java.util.Set;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.DiQuotaMaxDeltaUpdate;
import ru.yandex.qe.dispenser.api.v1.DiQuotaMaxUpdate;
import ru.yandex.qe.dispenser.api.v1.DiQuotaMaxUpdateBase;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;

public class QuotaMaxUpdate {
    @Nullable
    private final Long max;
    @Nullable
    private final Long ownMax;
    @NotNull
    private final Quota.Key quotaKey;

    private QuotaMaxUpdate(@Nullable final Long max, @Nullable final Long ownMax, @NotNull final Quota.Key quotaKey) {
        if (max == null && ownMax == null) {
            throw new IllegalArgumentException("Incorrect update for quota! Both 'max' and 'ownMax' fields are absent. " +
                    quotaKey.humanReadable());
        }
        this.max = max;
        this.ownMax = ownMax;
        this.quotaKey = quotaKey;
    }

    public static QuotaMaxUpdate from(@NotNull final DiQuotaMaxUpdate quotaMaxUpdate,
                                      @NotNull final Service service) {
        final Quota.Key quotaKey = quotaKey(quotaMaxUpdate, service);

        final DiUnit baseUnit = quotaKey.getSpec().getResource().getType().getBaseUnit();
        final Long max = Optional.ofNullable(quotaMaxUpdate.getMax()).map(baseUnit::convert).orElse(null);
        final Long ownMax = Optional.ofNullable(quotaMaxUpdate.getOwnMax()).map(baseUnit::convert).orElse(null);

        return new QuotaMaxUpdate(max, ownMax, quotaKey);
    }

    public static QuotaMaxUpdate from(@NotNull final DiQuotaMaxDeltaUpdate quotaMaxUpdate,
                                      @NotNull final Service service) {
        final Quota.Key quotaKey = quotaKey(quotaMaxUpdate, service);

        final DiUnit baseUnit = quotaKey.getSpec().getResource().getType().getBaseUnit();
        final Long max = Optional.ofNullable(quotaMaxUpdate.getMaxDelta()).map(delta -> delta.toSignedValue(baseUnit)).orElse(null);
        final Long ownMax = Optional.ofNullable(quotaMaxUpdate.getOwnMaxDelta()).map(delta -> delta.toSignedValue(baseUnit)).orElse(null);

        return new QuotaMaxUpdate(max, ownMax, quotaKey);
    }

    @NotNull
    private static Quota.Key quotaKey(@NotNull final DiQuotaMaxUpdateBase update, @NotNull final Service service) {
        final Project project = Hierarchy.get().getProjectReader().read(update.getProjectKey());
        final Resource resource = Hierarchy.get().getResourceReader().read(new Resource.Key(update.getResourceKey(), service));
        final QuotaSpec quotaSpec = Hierarchy.get().getQuotaSpecReader().read(resource, update.getQuotaSpecKey());
        final Set<Segment> segments = SegmentUtils.getCompleteSegmentSet(resource, update.getSegmentKeys());

        return new Quota.Key(quotaSpec, project, segments);
    }

    public static QuotaMaxUpdate from(@NotNull final Quota.Key quotaKey,
                                      @Nullable final Long max, @Nullable final Long ownMax) {
        return new QuotaMaxUpdate(max, ownMax, quotaKey);
    }

    @Nullable
    public Long getOwnMax() {
        return ownMax;
    }

    @Nullable
    public Long getMax() {
        return max;
    }

    @NotNull
    public Quota.Key getQuotaKey() {
        return quotaKey;
    }
}
