package ru.yandex.qe.dispenser.domain;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang3.builder.CompareToBuilder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.DiQuotaSpec;
import ru.yandex.qe.dispenser.api.v1.DiQuotaType;
import ru.yandex.qe.dispenser.domain.index.NormalizedPrimaryKeyBase;
import ru.yandex.qe.dispenser.domain.trans.QuotaTrans;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;

/**
 * Спецификация {@link Quota квоты}. Содержит квотируемый {@link Resource ресурс} и {@link DiQuotaType тип квотирования}.
 */
public final class QuotaSpec extends NormalizedPrimaryKeyBase<QuotaSpec.Key> {
    @NotNull
    private final String description;
    @NotNull
    private final DiQuotaType type;
    @NotNull
    private final List<QuotaTrans> transes;

    private QuotaSpec(@NotNull final Builder builder) {
        super(builder.key);
        this.description = ValidationUtils.requireNonNull(builder.description, "No description!");
        this.type = builder.type;
        this.transes = builder.transes;
    }

    @NotNull
    public String getPublicKey() {
        return getKey().getPublicKey();
    }

    @NotNull
    public Resource getResource() {
        return getKey().getResource();
    }

    @NotNull
    public String getDescription() {
        return description;
    }

    @NotNull
    public DiQuotaType getType() {
        return type;
    }

    @NotNull
    public List<QuotaTrans> getTranses() {
        return transes;
    }

    public @NotNull DiQuotaSpec toView() {
        return new DiQuotaSpec.Builder(getKey().getPublicKey())
                .resource(getKey().getResource().toView())
                .description(getDescription())
                .type(getType())
                .build();
    }

    public static final class Key implements Comparable<Key> {
        @NotNull
        private final String key;
        @NotNull
        private final Resource resource;

        public Key(@NotNull final String key, @NotNull final Resource resource) {
            this.key = key;
            this.resource = resource;
        }

        @NotNull
        public String getPublicKey() {
            return key;
        }

        @NotNull
        public Resource getResource() {
            return resource;
        }

        @Override
        public int compareTo(@NotNull final Key key) {
            return new CompareToBuilder()
                    .append(getResource(), key.getResource())
                    .append(getPublicKey(), key.getPublicKey())
                    .build();
        }

        @Override
        public boolean equals(@Nullable final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            return key.equals(((Key) o).key) && resource.equals(((Key) o).resource);
        }

        @Override
        public int hashCode() {
            return 31 * key.hashCode() + resource.hashCode();
        }

        @NotNull
        @Override
        public String toString() {
            return "DiQuotaSpec.Key{key='" + key + "\', resource.key=" + resource.getKey().getPublicKey() + '}';
        }
    }

    public static final class Builder {
        @NotNull
        private final Key key;
        @Nullable
        private String description;
        @NotNull
        private DiQuotaType type = DiQuotaType.ABSOLUTE;
        @NotNull
        private final List<QuotaTrans> transes = new ArrayList<>();

        public Builder(@NotNull final String key, @NotNull final Resource resource) {
            this.key = new Key(key, resource);
        }

        @NotNull
        public Builder description(@NotNull final String description) {
            this.description = description;
            return this;
        }

        @NotNull
        public Builder type(@NotNull final DiQuotaType type) {
            this.type = type;
            return this;
        }

        @NotNull
        public Builder addTranses(@NotNull final QuotaTrans... newTranses) {
            return addTranses(Arrays.asList(newTranses));
        }

        @NotNull
        public Builder addTranses(@NotNull final Iterable<QuotaTrans> transes) {
            transes.forEach(this.transes::add);
            return this;
        }

        public @NotNull QuotaSpec build() {
            return new QuotaSpec(this);
        }
    }
}
