package ru.yandex.qe.dispenser.domain;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.builder.CompareToBuilder;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.v1.DiSegment;
import ru.yandex.qe.dispenser.domain.index.NormalizedPrimaryKeyBase;

/**
 * Составная часть сегментированной квоты.
 * <p>
 * Сегмент может быть либо частью сегментации, либо агрегировать все сегменты сегментации ({@link Segment#totalOf(Segmentation)}).
 *
 * @see Segmentation
 * @see Quota
 */
@ParametersAreNonnullByDefault
public class Segment extends NormalizedPrimaryKeyBase<Segment.Key> {

    private static final String AGGREGATION_SEGMENT_KEY = "__aggregation__";

    private final String name;
    private final String description;
    private final Short priority;

    public Segment(final Builder builder) {
        super(builder.key);
        name = builder.name;
        description = builder.description;
        priority = builder.priority;
    }

    /**
     * Создает агрегирующий сегмент для заданной сегментации.
     */
    public static Segment totalOf(final Segmentation segmentation) {
        return new Builder(AGGREGATION_SEGMENT_KEY, segmentation)
                .build();
    }

    @NotNull
    public String getPublicKey() {
        return getKey().getPublicKey();
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public Short getPriority() {
        return priority;
    }

    public Segmentation getSegmentation() {
        return getKey().getSegmentation();
    }

    public boolean isAggregationSegment() {
        return getKey().getPublicKey().equals(AGGREGATION_SEGMENT_KEY);
    }

    public DiSegment toView() {
        return new DiSegment.Builder(getKey().getPublicKey())
                .inSegmentation(getKey().getSegmentation().toView())
                .withName(name)
                .withDescription(description)
                .withPriority(priority)
                .build();
    }

    @Override
    public String toString() {
        return getKey().getPublicKey();
    }

    public static final class Key implements Comparable<Segment.Key> {

        private final String key;
        private final Segmentation segmentation;

        public Key(final String key, final Segmentation segmentation) {
            this.key = key;
            this.segmentation = segmentation;
        }

        @NotNull
        public String getPublicKey() {
            return key;
        }

        @NotNull
        public Segmentation getSegmentation() {
            return segmentation;
        }

        @Override
        public int compareTo(final Segment.Key key) {
            return new CompareToBuilder()
                    .append(getSegmentation(), key.getSegmentation())
                    .append(getPublicKey(), key.getPublicKey())
                    .build();
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Key key1 = (Key) o;
            return Objects.equals(key, key1.key) &&
                    Objects.equals(segmentation, key1.segmentation);
        }

        @Override
        public int hashCode() {
            return Objects.hash(key, segmentation);
        }

        @NotNull
        @Override
        public String toString() {
            return "Key{key='" + key + "', segmentation.key=" + segmentation.getKey() + "}";
        }
    }

    public static final class Builder implements DtoBuilder<Segment> {

        private final Key key;
        private String name;
        private String description;
        private Short priority;

        public Builder(final String key, final Segmentation segmentation) {
            this.key = new Key(key, segmentation);
        }

        public Builder(final Key key) {
            this.key = key;
        }

        public Builder name(final String name) {
            this.name = name;
            return this;
        }

        public Builder description(final String description) {
            this.description = description;
            return this;
        }

        public Builder priority(final Short priority) {
            this.priority = priority;
            return this;
        }

        @NotNull
        @Override
        public Segment build() {
            return new Segment(this);
        }
    }

}
